package app.pivo.android.micsdk.controller.ble

import android.bluetooth.BluetoothGattCharacteristic
import android.bluetooth.BluetoothGattService
import android.content.Context
import android.os.ParcelUuid
import android.util.Log
import app.pivo.android.micsdk.Const
import app.pivo.android.micsdk.events.HeadsetEvent
import app.pivo.android.micsdk.events.PivoEventBus
import app.pivo.android.micsdk.util.PivoMicDevice
import com.polidea.rxandroidble.*
import com.polidea.rxandroidble.scan.ScanFilter
import com.polidea.rxandroidble.scan.ScanResult
import com.polidea.rxandroidble.scan.ScanSettings
import com.polidea.rxandroidble.utils.ConnectionSharingAdapter
import rx.Observable
import rx.Subscription
import rx.android.schedulers.AndroidSchedulers
import rx.internal.util.SubscriptionList
import java.util.*

/**
 * Created by murodjon on 2021/01/27
 *
 * This [BluetoothControllerImpl] class is implementation of [BluetoothController] interface.
 *
 * It is responsible for initiating connection, write/read bluetooth operations and connection failure.
 */
internal class BluetoothControllerImpl constructor(
    private val context: Context,
    private val callback: BluetoothControllerCallback
) : BluetoothController {

    private val TAG = this.javaClass.simpleName

    private var bleDevice: RxBleDevice? = null
    private var _bleClient: RxBleClient? = null
    private val bleClient get() = _bleClient!!
    private var bleConnectionObservable: Observable<RxBleConnection>? = null

    private var macAddress: String? = null
    private var headset: PivoMicDevice?=null

    init {
        createBleClient()
    }

    override fun scan() {
        // if the app is discovering now, return
        if (isScanning()) return

        // disconnect first, clear the results and unsubscribe, if the phone's connected to the device
        unsubscribe()

        // start scanning
        createScanningSubscriber()
    }

    override fun connectTo(device: PivoMicDevice) {
        this.headset = device
        // set mac address of  to be connected headset
        createBleDevice(device.getMacAddress())

        // unsubscribe subscriptions and clear them if they're used earlier
        unsubscribe()

        // connect to the headset
        connect()
    }

    override fun getHeadset(): PivoMicDevice? {
        return if (isConnected()){
            headset
        }else {
            null
        }
    }

    override fun isConnected(): Boolean {
        return bleDevice != null &&
                bleDevice!!.connectionState == RxBleConnection.RxBleConnectionState.CONNECTED
                && bleConnectionObservable != null
    }

    override fun cancelScan() {
        unsubscribe()
    }

    override fun disconnect() {
        // unsubscribe subscriptions and clear them if they're used earlier
        unsubscribe()

        // post connection failure
        postConnectionFailure()
    }

    override fun write(bytes: ByteArray) {
        if (isConnected()) {
            val subscription =
                bleConnectionObservable!!
                    .flatMap { rxBleConnection: RxBleConnection ->
                        rxBleConnection.writeCharacteristic(
                            Const.wUUID,// set up write characteristic
                            bytes
                        )
                    }
                    .observeOn(AndroidSchedulers.mainThread())
                    .subscribe(
                        { bytes: Any? -> onWriteSuccess(bytes) }
                    ) { throwable: Throwable? -> onWriteFailure(throwable) }
            operationSubscriptionList.add(subscription)
        } else {
            postConnectionFailure()
        }
    }

    /**
     * [createBleClient] is used to create bleClient using application context.
     * @param context is an application or an activity context
     */
    private fun createBleClient() {
        _bleClient = RxBleClient.create(context)
    }

    /**
     * [createBleDevice] is used to create bleDevice using mac address of bluetooth device
     * @param macAddress is address of bluetooth device
     */
    private fun createBleDevice(macAddress: String) {
        this.macAddress = macAddress
        bleDevice = bleClient.getBleDevice(macAddress)
    }

    /**
     * This [getBleConnectionObservable] gets an observable for [RxBleConnection].
     *
     * [RxBleConnection] is, the BLE connection handle, supporting GATT operations. Operations are enqueued and the library makes sure that they are not
     * executed in the same time within the client instance.
     */
    private fun getBleConnectionObservable(): Observable<RxBleConnection>? {
        return bleDevice
            ?.establishConnection(false)
            ?.doOnSubscribe({/*update UI*/ })
            ?.compose(ConnectionSharingAdapter())
    }

    /**
     * scanning subscription
     */
    private var scanSubscription: Subscription? = null

    /**
     * read, write, notify subscription list
     */
    private var subscriptionList = SubscriptionList()

    /**
     * all executed command subscription list
     */
    private var operationSubscriptionList: Queue<Subscription> = LinkedList()

    /**
     * clear the results and unsubscribe
     */
    private fun unsubscribe() {
        unsubscribeScanning()

        // remove all the subscription
        subscriptionList.unsubscribe()
        // refresh the list, because it's one time use only
        subscriptionList = SubscriptionList()

        operationSubscriptionList.forEach { subscription -> subscription.unsubscribe() }
        operationSubscriptionList = LinkedList()
    }

    /**
     * [createScanningSubscriber] is used to create subscription for scanning.
     */
    private fun createScanningSubscriber() {
        scanSubscription = bleClient
            .scanBleDevices(
                ScanSettings.Builder()
                    .build(),
                ScanFilter.Builder()// add filter to filter out Pivo devices using serviceUUID
                    .setServiceUuid(ParcelUuid(Const.serviceUUID))
                    .build()
            )
            ?.observeOn(AndroidSchedulers.mainThread())
            ?.doOnUnsubscribe { unsubscribe() }
            ?.subscribe(
                { result: ScanResult? -> result?.apply { addDevice(this) } },
                { throwable: Throwable? -> Log.e(TAG, "NotFound") })
    }

    /**
     * Unsubscribe scanning and make it null
     */
    private fun unsubscribeScanning() {
        scanSubscription?.unsubscribe()
        scanSubscription = null
    }

    /**
     * To check if the app is discovering devices
     */
    private fun isScanning(): Boolean {
        return scanSubscription != null
    }

    /**
     * notify scanned devices through [PivoEventBus]
     */
    private fun addDevice(result: ScanResult) {
        if (result.bleDevice != null) {
            PivoEventBus.publish(
                PivoEventBus.SCAN_DEVICE, HeadsetEvent.Scanning(
                    PivoMicDevice(
                        result.bleDevice.name,
                        result.bleDevice.macAddress
                    )
                )
            )
        }
    }

    /**
     * prepare connecting to headset device
     */
    private fun connect() {
        bleConnectionObservable = getBleConnectionObservable()
        val subscription = bleConnectionObservable?.subscribe(
            this::onConnectionReceived,
            this::onConnectionFailure
        )
        // add connection received and failure to subscription list
        subscriptionList.add(subscription)
    }

    private fun onConnectionFailure(throwable: Throwable?) {
        postConnectionFailure()
    }

    private fun onConnectionReceived(connection: RxBleConnection) {
        val serviceSubscription = connection.discoverServices()
            .observeOn(AndroidSchedulers.mainThread())
            .doOnUnsubscribe { this.onProgress() }
            .subscribe({ services: RxBleDeviceServices? ->
                services?.apply { setupConnection(this) }
            }) { throwable: Throwable? -> onConnectionFailure(throwable) }
        subscriptionList.add(serviceSubscription)
    }

    private fun onProgress() {
        Log.d(TAG, "onProgress")
    }

    private fun onNotificationSetupFailure(throwable: Throwable?) {
        Log.e(TAG, "onNotificationSetup failed: $throwable")
    }

    private fun onWriteSuccess(any: Any?) {
        // unsubscribe and remove subscription after writing successfully
        operationSubscriptionList.poll()?.unsubscribe()
    }

    private fun onWriteFailure(throwable: Throwable?) {
        Log.e(TAG, "onWriteFailed: $throwable")
        postConnectionFailure()
    }

    private fun setupConnection(services: RxBleDeviceServices) {
        var mBTService: BluetoothGattService? = null
        for (service in services.bluetoothGattServices) {
            if (service.uuid == Const.serviceUUID) {
                mBTService = service
                break
            }
        }
        if (mBTService == null) {
            postConnectionFailure()
            return
        }
        if (mBTService.uuid == null) {
            postConnectionFailure()
            return
        }
        // setup notification characteristic
        setupNotificationCharacteristic()
    }

    /**
     * [setupNotificationCharacteristic] is used to set up notification channel
     */
    private fun setupNotificationCharacteristic() {
        if (isConnected()) {
            val subscription = bleConnectionObservable
                ?.flatMap { rxBleConnection: RxBleConnection ->
                    rxBleConnection.setupNotification(// set up notification channel
                        Const.nUUID,
                        NotificationSetupMode.COMPAT
                    )
                }
                ?.doOnNext { postConnectionReceived() }// notification has been set up
                ?.flatMap { notificationObservable ->
                    notificationObservable
                }// Notification has been set up, now observe value changes
                ?.observeOn(AndroidSchedulers.mainThread())// observe changes on main thread
                ?.subscribe({ this.onNotificationReceived(it) })
                { throwable: Throwable? -> // handle an error here
                    onNotificationSetupFailure(throwable)
                }
            subscriptionList.add(subscription)
        }
    }

    private fun onNotificationReceived(bytes: ByteArray?) {
        bytes?.apply {
            callback.onNotificationReceived(bytes)
        }
    }

    private fun postConnectionFailure() {
        callback.onConnectionFailed()
    }

    private fun postConnectionReceived() {
        callback.onConnectionEstablished()
        // After a device is connected, be certain that you unsubscribe scanning.
        // Since it reduces the bandwidth available for any existing connections.
        unsubscribeScanning()
    }

    private fun isCharacteristicNotifiable(characteristic: BluetoothGattCharacteristic): Boolean {
        return characteristic.properties and BluetoothGattCharacteristic.PROPERTY_NOTIFY != 0
    }

    private fun isCharacteristicReadable(characteristic: BluetoothGattCharacteristic): Boolean {
        return characteristic.properties and BluetoothGattCharacteristic.PROPERTY_READ != 0
    }

    private fun isCharacteristicWriteable(characteristic: BluetoothGattCharacteristic): Boolean {
        return characteristic.properties and (BluetoothGattCharacteristic.PROPERTY_WRITE
                or BluetoothGattCharacteristic.PROPERTY_WRITE_NO_RESPONSE) != 0
    }
}