package app.pivo.android.micsdk.controller.cmd.parser

import android.util.Log
import app.pivo.android.micsdk.util.*

/**
 * Created by murodjon on 2021/01/26
 *
 * This is the implementation of[ReplyChecker] interface. And It has all necessary functions
 * for handling replies.
 */
class ReplyCheckerImpl(private val bytes: ByteArray) : ReplyChecker {

    init {
        Log.e("TTT", "convertHex: ${convertHexToString(bytes)}")
    }

    override fun getBatteryLevel(): Int? {
        return if (bytes.size == 6 && isBattery(bytes[2]) && isFromMCU(bytes[4])) {
            bytes[5].toInt()
        } else null
    }

    override fun getDeviceInfo(): DeviceInfo? {
        return if (bytes.size == 6 && isDeviceInfoCheckerReply(bytes[2]) && isFromMCU(bytes[4])) {
            val model = if (bytes[3] == 0xB0.toByte()) Model.PIVO_MIC else Model.OTHER
            DeviceInfo(model, bytes[5].toInt())
        } else {
            null
        }
    }

    override fun getDeviceStatus(): DeviceStatus? {
        return if (bytes.size == 11 && isFromMCU(bytes[4])) {
            DeviceStatus(
                led = getLEDType(bytes[5]),
                noise = getNoiseType(bytes[6]),
                sound = getSoundType(bytes[7]),
                micMode = getMicMode(bytes[8]),
                bypass = getBypassStatus(bytes[9]),
                earphoneStatus = getEarphoneStatus(bytes[10]),
                data = convertHexToString(bytes)
            )
        } else {
            null
        }
    }

    override fun getBypassStatus(): Bypass? {
        return if (bytes.size == 6 && isBypass(bytes[2]) && isFromMCU(bytes[4])) {
            getBypassStatus(bytes[5])
        } else {
            null
        }
    }

    override fun getLED(): LED? {
        return if (bytes.size == 6 && isLEDReply(bytes[2]) && isFromMCU(bytes[4])) {
            getLEDType(bytes[5])
        } else {
            null
        }
    }

    override fun getMicMode(): MicMode? {
        return if (bytes.size == 6 && isMicModeReply(bytes[2]) && isFromMCU(bytes[4])) {
            getMicMode(bytes[5])
        } else {
            null
        }
    }

    override fun getNoise(): Noise? {
        return if (bytes.size == 6 && isNoiseReply(bytes[2]) && isFromMCU(bytes[4])) {
            getNoiseType(bytes[5])
        } else {
            null
        }
    }

    override fun getSound(): Sound? {
        return if (bytes.size == 6 && isSoundReply(bytes[2]) && isFromMCU(bytes[4])) {
            getSoundType(bytes[5])
        } else {
            null
        }
    }

    private fun getSoundType(byte: Byte): Sound {
        return when (byte) {
            0x01.toByte() -> Sound.NORMAL
            0x02.toByte() -> Sound.BASS
            0x03.toByte() -> Sound.SMOOTH
            0x04.toByte() -> Sound.VIBRANT
            0x05.toByte() -> Sound.VOICE
            else -> Sound.TREBLE
        }
    }

    private fun getMicMode(byte: Byte): MicMode {
        return when (byte) {
            0x01.toByte() -> MicMode.DEFAULT
            0x02.toByte() -> MicMode.INTERNAL_MIC_ONLY
            0x03.toByte() -> MicMode.EXTERNAL_MIC_ONLY
            else -> MicMode.ALL_MIC_OFF
        }
    }

    private fun getLEDType(byte: Byte): LED {
        return when (byte) {
            0x01.toByte() -> LED.BLINKING_WHITE
            0x02.toByte() -> LED.WHITE
            0x03.toByte() -> LED.BLINKING_BLUE
            0x04.toByte() -> LED.BLUE
            0x05.toByte() -> LED.BLINKING_RED
            0x06.toByte() -> LED.RED
            else -> LED.OFF
        }
    }

    private fun getNoiseType(byte: Byte): Noise {
        return if (byte == 0x00.toByte()) {
            Noise.DISABLED
        } else {
            Noise.ENABLED
        }
    }

    override fun getMacAddress(): String? {
        return if (bytes.size >= 9 && isMacAddressReply(bytes[2])) {
            convertHexToFormattedMacAddress(bytes.copyOfRange(3, 9))
        } else null
    }

    override fun getSerialNum(): String? {
        return if (bytes.size >= 15 && isSerialNumReply(bytes[2])) {
            val sizeOfBytes = bytes.size
            convertHexToSerialNumber(bytes.copyOfRange(3, sizeOfBytes))
        } else null
    }

    override fun getName(): String? {
        return if (bytes.size >= 6 && isNameChangeReply(bytes[2]) && isFromMCU(bytes[4])) {
            convertHexToText(bytes.copyOfRange(5, bytes.size))
        } else null
    }

    override fun getEventType(): HeadsetNotification? {
        return if (bytes.size == 6 && isHeadsetSystemReply(bytes[2]) && isFromMCU(bytes[4])) {
            when {
                isMainBtnReply(bytes) -> {
                    HeadsetNotification.MAIN
                }
                isPlusBtnReply(bytes) -> {
                    HeadsetNotification.PLUS
                }
                isMinusBtnReply(bytes) -> {
                    HeadsetNotification.MINUS
                }
                isDeviceStatusReply(bytes) -> {
                    HeadsetNotification.DEVICE_STATUS
                }
                isDeviceConnectivityReply(bytes) -> {
                    HeadsetNotification.CONNECTIVITY_STATUS
                }
                else -> null
            }
        } else null
    }

    fun getBtnState(): Int {
        return if (bytes.size == 6) bytes[3].toInt() else 0
    }

    fun getDeviceStatusChange(): Status? {
        return when {
            bytes.size < 6 -> {
                null
            }
            bytes[3] == 0x10.toByte() -> Status.GOING_TO_SLEEP
            else -> Status.GOING_TO_BE_OFF
        }
    }

    private fun getBypassStatus(byte: Byte): Bypass? {
        return when (byte) {
            0xA0.toByte() -> {
                Bypass.OFF
            }
            0xA1.toByte() -> {
                Bypass.ON
            }
            else -> {
                null
            }
        }
    }

    fun getEarphoneStatus(byte: Byte): EarphoneStatus? {
        return when (byte) {
            0x00.toByte() -> EarphoneStatus.NOT_PLUGGED
            0x01.toByte() -> EarphoneStatus.PLUGGED_WITH_MIC
            0x02.toByte() -> EarphoneStatus.PLUGGED_WITHOUT_MIC
            else -> null
        }
    }

    private fun convertHexToString(data: ByteArray): String {
        val stringBuilder = StringBuilder(data.size)
        for (byteChar in data) stringBuilder.append(String.format("%02X ", byteChar))
        return stringBuilder.toString()
    }

    private fun convertHexToFormattedMacAddress(data: ByteArray): String {
        return convertHexToString(data)
    }

    // TODO: 2021/01/27 Need to check with Chinese manufacturer 
    private fun convertHexToSerialNumber(data: ByteArray): String {
        val stringBuilder = StringBuilder(data.size)
        data.copyOfRange(0, 6).forEachIndexed { i, v -> stringBuilder.append(if (i == 3) v.toInt() else v.toChar()) }
        data.copyOfRange(9, 12).forEachIndexed { index, byte -> stringBuilder.append(String.format("%02X", byte)) }
        return stringBuilder.toString()
    }

    private fun convertHexToText(data: ByteArray): String {
        val stringBuilder = StringBuilder(data.size)
        data.forEachIndexed { i, v -> stringBuilder.append(v.toChar()) }
        return stringBuilder.toString()
    }

    fun getCommandType(bytes: ByteArray): Command? {
        return if (bytes.size < 3) {
            null
        } else {
            when {
                isBattery(bytes[2]) -> Command.BATTERY
                isDeviceInfoCheckerReply(bytes[2]) -> Command.DEVICE_INFO
                isLEDReply(bytes[2]) -> Command.LED
                isNoiseReply(bytes[2]) -> Command.NOISE
                isSoundReply(bytes[2]) -> Command.SOUND
                isNameChangeReply(bytes[2]) -> Command.CHANGE_NAME
                isMacAddressReply(bytes[2]) -> Command.MAC_ADDRESS
                isSerialNumReply(bytes[2]) -> Command.SERIAL_NUM
                isHeadsetSystemReply(bytes[2]) -> Command.HEADSET_NOTIFICATION
                isMicStatusReply(bytes[2]) -> Command.MIC_STATUS
                isMicModeReply(bytes[2]) -> Command.MIC_MODE
                isBypass(bytes[2]) -> Command.BYPASS_STATUS
                else -> null
            }
        }
    }

    private fun isFromMCU(byte: Byte) = byte == 0x01.toByte()

    private fun isBattery(byte: Byte): Boolean {
        return byte == 0x04.toByte()
    }

    private fun isBypass(byte: Byte): Boolean {
        return byte == 0x14.toByte()
    }

    private fun isDeviceInfoCheckerReply(byte: Byte): Boolean {
        return byte == 0x03.toByte()
    }

    private fun isMicModeReply(byte: Byte): Boolean {
        return byte == 0x13.toByte()
    }

    private fun isLEDReply(byte: Byte): Boolean {
        return byte == 0x10.toByte()
    }

    private fun isNoiseReply(byte: Byte): Boolean {
        return byte == 0x11.toByte()
    }

    private fun isSoundReply(byte: Byte): Boolean {
        return byte == 0x12.toByte()
    }

    private fun isMacAddressReply(byte: Byte): Boolean {
        return byte == 0x1B.toByte()
    }

    private fun isSerialNumReply(byte: Byte): Boolean {
        return byte == 0x1C.toByte()
    }

    private fun isHeadsetSystemReply(byte: Byte): Boolean {
        return byte == 0x20.toByte()
    }

    private fun isMainBtnReply(bytes: ByteArray): Boolean {
        return bytes[5] == 0x01.toByte()
    }

    private fun isPlusBtnReply(bytes: ByteArray): Boolean {
        return bytes[5] == 0x11.toByte()
    }

    private fun isMinusBtnReply(bytes: ByteArray): Boolean {
        return bytes[5] == 0x21.toByte()
    }

    private fun isDeviceStatusReply(bytes: ByteArray): Boolean {
        return bytes[5] == 0x00.toByte()
    }

    private fun isDeviceConnectivityReply(bytes: ByteArray): Boolean {
        return bytes[5] == 0x30.toByte()
    }

    private fun isNameChangeReply(byte: Byte): Boolean {
        return byte == 0x7F.toByte()
    }

    private fun isMicStatusReply(byte: Byte): Boolean {
        return byte == 0x1D.toByte()
    }
}