package app.pivo.android.micsdk.events

import android.util.Log
import android.util.SparseArray
import androidx.annotation.IntDef
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.disposables.CompositeDisposable
import io.reactivex.functions.Consumer
import io.reactivex.subjects.PublishSubject
import java.util.*

/**
 * Created by murodjon on 2021/01/19
 */
/**
 * PivoEventBus allows to listen and send Pivo events [PivoEvent] in different components of the application.
 */
object PivoEventBus {
    private val sSubjectMap = SparseArray<PublishSubject<Any>>()
    private val sSubscriptionsMap: MutableMap<Any, CompositeDisposable> = HashMap()

    /**
     * Invalid reply
     */
    const val INVALID_REPLY = -1

    /**
     * Connection completed subject.
     */
    const val CONNECTION_COMPLETED = 0

    /**
     * Connection failed or an app is disconnected from Pivo.
     */
    const val CONNECTION_FAILURE = 1

    /**
     * Scan device subject.
     */
    const val SCAN_DEVICE = 2

    /**
     * Remote controller subject.
     */
    const val SYSTEM_NOTIFICATION = 3

    /**
     * Name is changed subject.
     */
    const val NAME_CHANGED = 4

    /**
     * MacAddress subject
     */
    const val MAC_ADDRESS = 5

    /**
     * Serial number subject
     */
    const val SERIAL_NUMBER = 6

    /**
     * Version subject
     */
    const val VERSION_NOTIFICATION = 7

    /**
     * Control subject
     */
    const val CONTROL_NOTIFICATION = 8

    /**
     * Pivo notification subject
     */
    const val PIVO_NOTIFICATION = 9

    /**
     * Pivo Mic Status subject
     */
    const val PIVO_HEADSET_STATUS = 10

    /**
     * Pivo Headset bypass subject
     */
    const val PIVO_HEADSET_BYPASS = 11


    /**
     * Get the subject or create it if it's not already in memory.
     */
    private fun getSubject(@Subject subjectCode: Int): PublishSubject<Any> {
        var subject = sSubjectMap[subjectCode]
        if (subject == null) {
            subject = PublishSubject.create()
            subject.subscribeOn(AndroidSchedulers.mainThread())
            sSubjectMap.put(subjectCode, subject)
        }
        return subject
    }

    /**
     * Get the CompositeDisposable or create it if it's not already in memory.
     */
    private fun getCompositeDisposable(`object`: String): CompositeDisposable {
        var compositeDisposable = sSubscriptionsMap[`object`]
        if (compositeDisposable == null) {
            compositeDisposable = CompositeDisposable()
            sSubscriptionsMap[`object`] = compositeDisposable
        }
        return compositeDisposable
    }

    /**
     * Subscribe to the specified subject and listen for updates on that subject. Pass in an object to associate
     * your registration with, so that you can unsubscribe later.
     * <br></br><br></br>
     * **Note:** Make sure to call [PivoEventBus.unregister] to avoid memory leaks.
     */
    fun subscribe(@Subject subject: Int, lifecycle: String, action: Consumer<Any>) {
        val disposable = getSubject(subject).subscribe(action,
            { throwable: Throwable? ->
                Log.e(
                    "PivoEventBus",
                    "Exception"
                )
            }) //.subscribe(action);
        getCompositeDisposable(lifecycle).add(disposable)
    }

    /**
     * Unregisters this object from the bus, removing all subscriptions.
     * This should be called when the object is going to go out of memory.
     */
    fun unregister(lifecycle: Any) {
        //We have to remove the composition from the map, because once you dispose it can't be used anymore
        val compositeDisposable = sSubscriptionsMap.remove(lifecycle)
        compositeDisposable?.dispose()
    }

    /**
     * Publish an object to the specified subject for all subscribers of that subject.
     */
    fun publish(@Subject subject: Int, message: Any) {
        getSubject(subject).onNext(message)
    }

    @kotlin.annotation.Retention(AnnotationRetention.SOURCE)
    @IntDef(
        INVALID_REPLY,
        CONNECTION_COMPLETED,
        CONNECTION_FAILURE,
        SCAN_DEVICE,
        SYSTEM_NOTIFICATION,
        NAME_CHANGED,
        PIVO_NOTIFICATION,
        MAC_ADDRESS,
        SERIAL_NUMBER,
        VERSION_NOTIFICATION,
        CONTROL_NOTIFICATION,
        PIVO_HEADSET_STATUS,
        PIVO_HEADSET_BYPASS
    )
    internal annotation class Subject
}