package app.pivo.android.micsdk.controller.ble

import android.bluetooth.BluetoothGattService
import android.content.Context
import android.os.Build
import android.os.ParcelUuid
import android.util.Log
import app.pivo.android.micsdk.events.PivoEventBus
import app.pivo.android.micsdk.util.PivoDeviceUUID
import app.pivo.android.micsdk.util.PivoMicDevice
import com.jakewharton.rx.ReplayingShare
import com.polidea.rxandroidble2.*
import com.polidea.rxandroidble2.scan.ScanFilter
import com.polidea.rxandroidble2.scan.ScanResult
import com.polidea.rxandroidble2.scan.ScanSettings
import io.reactivex.Observable
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.disposables.CompositeDisposable
import io.reactivex.disposables.Disposable
import io.reactivex.subjects.PublishSubject
import java.util.*

/**
 * Created by murodjon on 2021/01/27
 *
 * This [BluetoothControllerImpl] class is implementation of [BluetoothController] interface.
 *
 * It is responsible for initiating connection, write/read bluetooth operations and connection failure.
 */
internal class BluetoothControllerImpl constructor(
    private val context: Context,
    private val callback: BluetoothControllerCallback,
    private val uuids: Map<UUID, PivoDeviceUUID>
) : BluetoothController {

    private val TAG = this.javaClass.simpleName

    private var bleDevice: RxBleDevice? = null
    private var _bleClient: RxBleClient? = null
    private val bleClient get() = _bleClient!!
    private var bleConnectionObservable: Observable<RxBleConnection>? = null
    private var connectionStateDisposable: Disposable? = null

    private val disconnectTriggerSubject = PublishSubject.create<Unit>()

    private var macAddress: String? = null
    private var pivoDevice: PivoMicDevice? = null
    private var pivoConnectionOnOffLastDevice: PivoMicDevice? = null

    private var podUUD: PivoDeviceUUID? = null

    //when connection failed, connection will try again.
    private var connectionRetryCount = 0

    init {
        createBleClient()
    }

    /**
     * This function[scan] is used to scan ble devices
     */
    override fun scan() {
        // if the app is discovering now, return
        if (isScanning()) {
            Log.d(TAG, "scan : return")
            return
        }

        Log.d(TAG, "scan")

        // disconnect first, clear the results and unsubscribe, if the phone's connected to the device
        unsubscribeScanning()

        // start scanning
        createScanningSubscriber()
    }

    /**
     * This function[connectTo] is used to connect to a scanned specific device
     */
    @Synchronized
    override fun connectTo(device: PivoMicDevice) {
        Log.d(TAG, "connectTo")

        this.pivoDevice = device

        // set mac address of  to be connected headset
        createBleDevice(device.getMacAddress())

        // unsubscribe subscriptions and clear them if they're used earlier
        unsubscribe()

        // connect to the headset
        connect()

        callback.onConnectedDevices(pivoDevice)
    }

    override fun getDevice(): PivoMicDevice? {
        return if (isConnected()) {
            pivoDevice
        } else {
            null
        }
    }

    /**
     * This function[isConnected] is used to check connectivity
     */
    override fun isConnected(): Boolean {
        return bleDevice != null &&
                bleDevice!!.connectionState == RxBleConnection.RxBleConnectionState.CONNECTED
                && bleConnectionObservable != null
    }

    /**
     * This function[cancelScan] is used to cancel scanning ble devices
     */
    override fun cancelScan() {
        unsubscribeScanning()
    }

    /**
     * This function[disconnect] is used to disconnect from a ble device
     */
    override fun disconnect() {
        Log.d(TAG, "disconnect")

        // unsubscribe subscriptions and clear them if they're used earlier
        unsubscribe()

        // post connection failure
        postConnectionFailure()
    }


    override fun setConnectionOnOffLastDevice(isOn: Boolean) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
            pivoDevice?.let {
                if (isOn && !isConnected() && pivoConnectionOnOffLastDevice != null) {
                    connect {
                        Log.d(TAG, "setConnectionOnOffLastDevice : connect fail... retry")
                        unsubscribe()
                        connect()
                    }
                    pivoConnectionOnOffLastDevice = null
                    Log.d(TAG, "setConnectionOnOffLastDevice : connect")
                } else if (!isOn && isConnected()) {
                    pivoConnectionOnOffLastDevice = it
                    disconnect()
                    Log.d(TAG, "setConnectionOnOffLastDevice : disconnect")
                } else {
                    Log.d(TAG, "setConnectionOnOffLastDevice : else")
                }
            }
        }
    }

    /**
     * This function[write] is used to write command bytes
     */
    override fun write(bytes: ByteArray) {
        if (isConnected()) {
            Log.d(TAG, "write: $bytes")
            val writeDisposable =
                bleConnectionObservable
                    ?.flatMapSingle { rxBleConnection: RxBleConnection ->
                        podUUD?.let {
                            rxBleConnection.writeCharacteristic(
                                it.writeUUID,// set up write characteristic
                                bytes
                            )
                        }
                    }
                    ?.observeOn(AndroidSchedulers.mainThread())
                    ?.subscribe(
                        { bytes: Any? -> onWriteSuccess(bytes) }
                    ) { throwable: Throwable? ->
                        onWriteFailure(throwable)
                        Log.e(TAG, "write: onWriteFailure : $throwable")
                    }

            writeDisposable?.let {
                writeOperationDisposables.add(it)
            }
        } else {
            Log.e(TAG, "write: isConnected == false")
            postConnectionFailure()
        }
    }

    /**
     * [createBleClient] is used to create bleClient using application context.
     * @param context is an application or an activity context
     */
    private fun createBleClient() {
        _bleClient = RxBleClient.create(context)
    }

    /**
     * [createBleDevice] is used to create bleDevice using mac address of bluetooth device
     * @param macAddress is address of bluetooth device
     */
    private fun createBleDevice(macAddress: String) {
        this.macAddress = macAddress
        bleDevice = bleClient.getBleDevice(macAddress)
        connectionStateDisposable = bleDevice
            ?.observeConnectionStateChanges()
            ?.subscribe(
                { connectionState ->
                    Log.d(TAG, "createBleDevice: $connectionState")
                }
            ) { throwable ->
                Log.e(TAG, "createBleDevice exception: $throwable")
            }
    }

    /**
     * This [prepareConnectionObservable] gets an observable for [RxBleConnection].
     *
     * [RxBleConnection] is, the BLE connection handle, supporting GATT operations. Operations are enqueued and the library makes sure that they are not
     * executed in the same time within the client instance.
     */
    private fun prepareConnectionObservable(): Observable<RxBleConnection>? =
        bleDevice
            ?.establishConnection(false)
            ?.takeUntil(disconnectTriggerSubject)
            ?.compose(ReplayingShare.instance())

    /**
     * scanning subscription
     */
    private var scanSubscription: Disposable? = null

    /**
     * read, write, notify subscription list
     */
    private var operationDisposable = CompositeDisposable()

    /**
     * all executed command subscription list
     */
    private var writeOperationDisposables: Queue<Disposable?> = LinkedList()

    /**
     * clear the results and unsubscribe
     */
    private fun unsubscribe() {
        unsubscribeScanning()

        // remove all the subscription
        operationDisposable.dispose()
        // refresh the list, because it's one time use only
        operationDisposable = CompositeDisposable()

        writeOperationDisposables.toTypedArray().forEach { subscription ->
            subscription?.dispose()
        }
        writeOperationDisposables = LinkedList()

        connectionStateDisposable?.dispose()
    }

    /**
     * [provideScanningFilters] is used to create filters for scanning PodX and pod devices
     */
    private fun provideScanningFilters(): List<ScanFilter> {
        val filters = mutableListOf<ScanFilter>()
        uuids.values.forEach {
            filters.add(
                ScanFilter.Builder()// add filter to filter out Pivo devices using serviceUUID
                    .setServiceUuid(ParcelUuid(it.serviceUUID))
                    .build()
            )
        }
        return filters
    }

    /**
     * [createScanningSubscriber] is used to create subscription for scanning.
     */
    private fun createScanningSubscriber() {
        scanSubscription = bleClient
            .scanBleDevices(
                ScanSettings.Builder()
                    .build(),
                *provideScanningFilters().toTypedArray()
            )
            ?.subscribe(
                { result: ScanResult? -> result?.apply { addDevice(this) } },
                { throwable: Throwable? -> Log.e(TAG, "createScanningSubscriber : $throwable") })
    }

    /**
     * Unsubscribe scanning and make it null
     */
    private fun unsubscribeScanning() {
        scanSubscription?.dispose()
        scanSubscription = null
    }

    /**
     * To check if the app is discovering devices
     */
    private fun isScanning(): Boolean {
        return scanSubscription != null
    }

    /**
     * notify scanned devices through [PivoEventBus]
     */
    private fun addDevice(result: ScanResult) {
        if (result.bleDevice != null) {
            callback.onAddDevice(
                PivoMicDevice(
                    name = result.bleDevice.name ?: "",
                    macAddress = result.bleDevice.macAddress
                )
            )
        }
    }

    /**
     * prepare connecting to headset device
     */
    private fun connect(connectionFailer: ((throwable: Throwable?) -> Unit)? = null) {
        bleConnectionObservable = prepareConnectionObservable()
        bleConnectionObservable
            ?.flatMapSingle { it.discoverServices() }
            ?.observeOn(AndroidSchedulers.mainThread())
            ?.doOnSubscribe { onProgress() }
            ?.subscribe(
                this::onConnectionReceived,
                connectionFailer ?: this::onConnectionFailure
            )?.let { operationDisposable.add(it) }
    }

    /**
     * This [onConnectionFailure] triggered if connection faails
     */
    private fun onConnectionFailure(throwable: Throwable?) {
        Log.e(TAG, "onConnectionFailure failed: $throwable")
        // if connection is failed, it requests connection again in tree times.
        if (connectionRetryCount < CONNECTION_RETRY_MAX_COUNT) {
            connectionRetryCount = connectionRetryCount + 1
            Log.d(TAG, "onConnectionFailure -- retry count : " + connectionRetryCount)
            pivoDevice?.let {
                connectTo(pivoDevice!!)
            }
            return
        }
        postConnectionFailure()
    }

    /**
     * This [postConnectionFailure] is triggered when connection failure occurs
     */
    private fun postConnectionFailure() {
        if (pivoConnectionOnOffLastDevice != null) {
            Log.e(TAG, "postConnectionFailure: return")
            return
        }

        Log.e(TAG, "postConnectionFailure: callback")

        callback.onConnectionFailed()
    }

    private fun onProgress() {
        Log.d(TAG, "onProgress")
    }


    /**
     * [onWriteSuccess] is used to inform if the write operation's successful
     */
    private fun onWriteSuccess(any: Any?) {
        // unsubscribe and remove subscription after writing successfully
        writeOperationDisposables.poll()?.dispose()
    }

    /**
     * [onWriteFailure] is used to notify if the write operation's failed
     */
    private fun onWriteFailure(throwable: Throwable?) {
        Log.e(TAG, "onWriteFailed: $throwable")
        postConnectionFailure()
    }

    /**
     * This [getConnectedGattService] function checks the connected
     * device by UUID and returns it.
     */
    private fun getConnectedGattService(services: RxBleDeviceServices): BluetoothGattService? {
        var mBTService: BluetoothGattService? = null
        for (service in services.bluetoothGattServices) {
            if (uuids.containsKey(service.uuid)) {
                mBTService = service
                podUUD = uuids[service.uuid]
                break
            }
        }
        return mBTService
    }

    /**
     * This [onConnectionReceived] is triggered if the connection is established
     */
    private fun onConnectionReceived(services: RxBleDeviceServices) {
        val mBTService = getConnectedGattService(services)
        if (mBTService == null || mBTService.uuid == null) {
            postConnectionFailure()
            return
        }
        // setup notification characteristic
        setupNotification()

        // if connection is completed, retry count should be reset.
        connectionRetryCount = 0
    }

    /**
     * [setupNotification] is used to set up notification channel
     */
    private fun setupNotification() {
        if (isConnected()) {
            bleConnectionObservable
                ?.flatMap {
                    podUUD?.let { it1 ->
                        it.setupNotification(
                            it1.notificationUUID,
                            NotificationSetupMode.QUICK_SETUP
                        )
                    }
                }
                ?.doOnNext { notificationHasBeenSetUp() }
                ?.flatMap { notificationObservable -> notificationObservable } // <-- Notification has been set up, now observe value changes.
                ?.subscribe(
                    { onNotificationReceived(it) },
                    { throwable -> onNotificationSetupFailure(throwable) })
                ?.let { operationDisposable.add(it) }
        }
    }

    /**
     * This [notificationHasBeenSetUp] is used to inform when the notification channel has been set up
     */
    private fun notificationHasBeenSetUp() {
        if (podUUD == null) {
            postConnectionFailure()
            return
        }
        callback.onConnectionEstablished()
        // After a device is connected, be certain that you unsubscribe scanning.
        // Since it reduces the bandwidth available for any existing connections.
        unsubscribeScanning()
    }

    /**
     * This function [onNotificationSetupFailure] triggered if the notification set up fails
     */
    private fun onNotificationSetupFailure(throwable: Throwable?) {
        Log.e(TAG, "onNotificationSetup failed: $throwable")
    }

    /**
     * This function [onNotificationReceived] is called when app receives any notification from BLE device
     */
    private fun onNotificationReceived(bytes: ByteArray?) {
        bytes?.apply {
            callback.onNotificationReceived(bytes)
        }
    }

    companion object {
        private const val TAG = "BluetoothController"

        //when connection failed, connection will try again.
        private const val CONNECTION_RETRY_MAX_COUNT = 3
    }
}