package app.pivo.android.micsdk.controller.cmd.parser

import android.util.Log
import app.pivo.android.micsdk.controller.ble.BluetoothControllerCallback
import app.pivo.android.micsdk.events.HeadsetEvent
import app.pivo.android.micsdk.events.PivoEventBus
import app.pivo.android.micsdk.util.Command
import app.pivo.android.micsdk.util.HeadsetNotification
import app.pivo.android.micsdk.util.PivoMicDevice

/**
 * Created by murodjon on 2021/01/22
 *
 * This [ReplyHandler] class has an implementation of [BluetoothControllerCallback].
 */
class ReplyHandler : BluetoothControllerCallback {

    private var macAddress: String? = null

    override fun onAddDevice(device: PivoMicDevice) {
        PivoEventBus.publish(
            PivoEventBus.SCAN_DEVICE, HeadsetEvent.Scanning(
                device
            )
        )
    }

    override fun onConnectionFailed() {
        PivoEventBus.publish(PivoEventBus.CONNECTION_FAILURE, HeadsetEvent.ConnectionFailure())
    }

    override fun onConnectionEstablished() {
        PivoEventBus.publish(PivoEventBus.CONNECTION_COMPLETED, HeadsetEvent.ConnectionComplete())
    }

    override fun onConnectedDevices(device: PivoMicDevice?) {
        this.macAddress = device?.getMacAddress()
    }

    override fun onNotificationReceived(bytes: ByteArray) {
        val replyChecker = ReplyCheckerImpl(bytes)

        Log.e("TTT", "notification: ${replyChecker.getCommandType(bytes)}")

        when (replyChecker.getCommandType(bytes)) {
            Command.DEVICE_INFO -> {
                (replyChecker as DeviceInfoChecker).getDeviceInfo()?.apply {
                    PivoEventBus.publish(
                        PivoEventBus.VERSION_NOTIFICATION,
                        HeadsetEvent.DeviceInfoReceived(this)
                    )
                }
            }
            Command.BATTERY -> {
                (replyChecker as BatteryLevelChecker).getBatteryLevel()?.apply {
                    PivoEventBus.publish(
                        PivoEventBus.PIVO_NOTIFICATION,
                        HeadsetEvent.BatteryChanged(this)
                    )
                }
            }
            Command.MIC_MODE -> {
                (replyChecker as MicSelectorChecker).getMicMode()?.apply {
                    PivoEventBus.publish(
                        PivoEventBus.CONTROL_NOTIFICATION,
                        HeadsetEvent.MicModeCntrlType(this)
                    )
                }
            }
            Command.LED -> {
                (replyChecker as LEDControlChecker).getLED()?.apply {
                    PivoEventBus.publish(
                        PivoEventBus.CONTROL_NOTIFICATION,
                        HeadsetEvent.LEDCntrlType(this)
                    )
                }
            }
            Command.NOISE -> {
                (replyChecker as NoiseCntrlChecker).getNoise()?.apply {
                    PivoEventBus.publish(
                        PivoEventBus.CONTROL_NOTIFICATION,
                        HeadsetEvent.NoiseCntrlType(this)
                    )
                }
            }
            Command.SOUND -> {
                (replyChecker as SoundCntrlChecker).getSound()?.apply {
                    PivoEventBus.publish(
                        PivoEventBus.CONTROL_NOTIFICATION,
                        HeadsetEvent.SoundCntrlType(this)
                    )
                }
            }
            Command.MAC_ADDRESS -> {
                (replyChecker as MacAddressChecker).getMacAddress()?.apply {
                    PivoEventBus.publish(PivoEventBus.MAC_ADDRESS, HeadsetEvent.MacAddress(this))
                }
            }
            Command.SERIAL_NUM -> {
                (replyChecker as SerialNumberChecker).getSerialNum(macAddress)?.apply {
                    PivoEventBus.publish(
                        PivoEventBus.SERIAL_NUMBER,
                        HeadsetEvent.SerialNumber(this)
                    )
                }
            }
            Command.CHANGE_NAME -> {
                (replyChecker as NameChangedChecker).getName()?.apply {
                    PivoEventBus.publish(PivoEventBus.NAME_CHANGED, HeadsetEvent.NameChanged(this))
                }
            }
            Command.HEADSET_NOTIFICATION -> {
                (replyChecker as HeadsetNotificationChecker).getEventType()?.apply {
                    when (this) {
                        HeadsetNotification.MAIN -> PivoEventBus.publish(
                            PivoEventBus.SYSTEM_NOTIFICATION,
                            HeadsetEvent.MainButton(replyChecker.getBtnState())
                        )
                        HeadsetNotification.PLUS -> PivoEventBus.publish(
                            PivoEventBus.SYSTEM_NOTIFICATION,
                            HeadsetEvent.PlusButton(replyChecker.getBtnState())
                        )
                        HeadsetNotification.MINUS -> PivoEventBus.publish(
                            PivoEventBus.SYSTEM_NOTIFICATION,
                            HeadsetEvent.MinusButton(replyChecker.getBtnState())
                        )
                        HeadsetNotification.DEVICE_STATUS -> replyChecker.getDeviceStatusChange()
                            ?.apply { HeadsetEvent.SystemStatus(this) }?.let {
                                PivoEventBus.publish(
                                    PivoEventBus.SYSTEM_NOTIFICATION,
                                    it
                                )
                            }
                        HeadsetNotification.CONNECTIVITY_STATUS -> replyChecker.getEarphoneStatus(
                            bytes[3]
                        )
                            ?.apply {
                                PivoEventBus.publish(
                                    PivoEventBus.SYSTEM_NOTIFICATION,
                                    HeadsetEvent.DeviceConnectivity(this)
                                )
                            }
                    }
                }
            }
            Command.MIC_STATUS -> {
                replyChecker.getDeviceStatus()?.let {
                    PivoEventBus.publish(
                        PivoEventBus.PIVO_HEADSET_STATUS,
                        it
                    )
                }
            }
            Command.BYPASS_STATUS -> {
                replyChecker.getBypassStatus()?.let {
                    PivoEventBus.publish(
                        PivoEventBus.PIVO_HEADSET_BYPASS,
                        HeadsetEvent.BypassStatus(it)
                    )
                }
            }
            else -> {//Command.WRONG_COMMAND

            }
        }
    }
}