package app.pivo.android.podsdk.cmd.builder.impl

import app.pivo.android.podsdk.auth.PodAuthenticator
import app.pivo.android.podsdk.cmd.builder.CommandBuilder
import app.pivo.android.podsdk.cmd.builder.ext.getHexValue
import app.pivo.android.podsdk.model.*
import app.pivo.android.podsdk.ota.FirmwareChunkProvider
import app.pivo.android.podsdk.util.HexUtil
import java.util.*

/**
 * Created by murodjon on 2022/01/11
 */
internal class PodMaxBuilderImpl : CommandBuilder {

    override fun getDeviceInfoBytes(): ByteArray {
        return byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x42, 0x01, 0x01, 0x00, 0x00)
    }

    override fun getTimeoutBytes(time: Int): ByteArray {
        return if (time % 10 == 0 && time >= 0 && time <= 2550) {
            byteArrayOf(
                0x05,
                0xAF.toByte(),
                0x05,
                0x42.toByte(),
                0x02,
                0x01,
                (time / 10).toByte(),
                0x00
            )
        } else {
            byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x42.toByte(), 0x02, 0x01, 0x00, 0x00)
        }
    }

    override fun getBatteryLevelBytes(): ByteArray =
        byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x42, 0x03, 0x01, 0x00, 0x00)

    override fun getSoundBytes(sound: SoundType): ByteArray =
        byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x42, 0x04, 0x01, sound.getHexValue(), 0x00)

    override fun getChangeNameBytes(name: String): ByteArray? {
        val length = name.length
        if (length == 0) return null
        // final byte array size for changing name is 5+(length of name)
        val bytes = ByteArray(length + 6)
        bytes[0] = 0x05.toByte()
        bytes[1] = 0xAF.toByte()
        bytes[2] = (length + 3).toByte()
        bytes[3] = 0x49.toByte()
        bytes[4] = 0x7F.toByte()
        bytes[5] = 0x01.toByte()
        name.forEachIndexed { i, v ->
            bytes[6 + i] = name[i].code.toByte()
        }
        return bytes
    }

    override fun getMacAddressBytes(): ByteArray =
        byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x49, 0x1B, 0x01, 0x00, 0x00)

    override fun getSerialNumberBytes(): ByteArray =
        byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x49, 0x1C, 0x01, 0x00, 0x00)

    override fun getStopBytes() =
        byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x4D, 0x00, 0x01, 0x00, 0x00)

    override fun getMoveHorizontallyBytes(
        version: Int,
        direction: HorizontalDirection,
        speed: Speed,
        degree: Int
    ): ByteArray? {
        val directionByte: Byte = if (direction == HorizontalDirection.LEFT)
            if (speed.speedUnit == SpeedUnit.SEC_PER_ROUND) 0x21 else 0x61
        else if (speed.speedUnit == SpeedUnit.SEC_PER_ROUND) 0x20 else 0x60

        val speedBytes = HexUtil.convertSpeedToHexBytes(speed.value) ?: return null

        val rotationBytes = HexUtil.convertHorizontalDegreeToHexByte(degree) ?: return null

        return byteArrayOf(
            0x05, 0xAF.toByte(), 0x07, 0x4D, /* direction */directionByte, 0x01,
            /* speed1 */speedBytes[0], /* speed1 */speedBytes[1],
            /* degree1 */rotationBytes[0], /* degree1 */rotationBytes[1]
        )
    }

    override fun getHorizontalPositionBytes() =
        byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x4D, 0x22, 0x01, 0x00, 0x00)

    override fun getGoToHorizontalPositionBytes(speed: Speed, degree: Int): ByteArray? {
        val speedBytes = HexUtil.convertSpeedToHexBytes(speed.value) ?: return null
        val rotationBytes = HexUtil.convertHorizontalDegreeToHexByte(degree) ?: return null
        val speedUnitByte: Byte = if (speed.speedUnit == SpeedUnit.SEC_PER_ROUND) 0x23 else 0x63

        return byteArrayOf(
            0x05, 0xAF.toByte(), 0x07, 0x4D, speedUnitByte, 0x01,
            /* speed1 */speedBytes[0], /* speed1 */speedBytes[1],
            /* degree1 */rotationBytes[0], /* degree1 */rotationBytes[1]
        )
    }

    override fun getSetHorizontalSpeedBytes(version: Int, speed: Speed): ByteArray? {
        val speedBytes = HexUtil.convertSpeedToHexBytes(speed.value) ?: return null
        val speedUnitByte: Byte = if (speed.speedUnit == SpeedUnit.SEC_PER_ROUND) 0x24 else 0x64

        return byteArrayOf(
            0x05, 0xAF.toByte(), 0x05, 0x4D, speedUnitByte, 0x01,
            /* speed1 */speedBytes[0], /* speed1 */speedBytes[1]
        )
    }

    override fun getResetPositionBytes() =
        byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x4D, 0x29, 0x01, 0x00, 0x00)

    override fun getNotifierSwitcherBytes(notifierEnabled: Boolean) =
        byteArrayOf(
            0x05,
            0xAF.toByte(),
            0x05,
            0x4D,
            0xF1.toByte(),
            0x01,
            if (notifierEnabled) 0xA1.toByte() else 0xA0.toByte(),
            0x00
        )

    /**
     * This function returns remote controller switcher bytes according to
     * @param remoteEnabled is true when remote controller and pod communicates
     */
    override fun getRemoteSwitcherBytes(remoteEnabled: Boolean) =
        byteArrayOf(
            0x05,
            0xAF.toByte(),
            0x05,
            0x52,
            0x01,
            0x01,
            if (remoteEnabled) 0xA1.toByte() else 0xA0.toByte(),
            0x00
        )

    /**
     * Gets start pairing bytes
     */
    override fun getStartPairingBytes() =
        byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x52, 0x02, 0x01, 0x00, 0x00)

    override fun getCancelPairingBytes() =
        byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x52, 0x03, 0x01, 0x00, 0x00)

    override fun getSetLEDColorBytes(colors: List<LEDColor>, brightness: Int): ByteArray? {
        if (colors.size != 12 && brightness !in 0..100) return null
        val data = byteArrayOf(
            0x05,
            0xAF.toByte(),
            0xF,
            0x4C,
            0x40,
            0x01,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            brightness.toByte()
        )
        colors.forEachIndexed { index, ledColor -> data[index + 6] = ledColor.getValue() }
        return data
    }

    override fun getRetrieveLEDColorBytes() =
        byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x4C, 0x41, 0x01, 0x00, 0x00)

    override fun getRunLightPresetBytes(patternType: LightPatternType, level: Int): ByteArray? {
        val patternByte: Byte = when (patternType) {
            LightPatternType.PatternA -> 0x01
            LightPatternType.PatternB -> 0x02
            LightPatternType.PatternC -> 0x03
            LightPatternType.PatternD -> 0x04
            LightPatternType.PatternE -> 0x05
            LightPatternType.PatternF -> 0x06
            LightPatternType.PatternG -> 0x07
            LightPatternType.PatternH -> 0x08
        }
        return byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x4C, 0x21, 0x01, patternByte, level.toByte())
    }

    override fun getStopLightPresetBytes() =
        byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x4C, 0x20, 0x01, 0x00, 0x00)


    override fun getOTAInitiationBytes(): ByteArray {
        return byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x55, 0x8F.toByte(), 0x01, 0x00, 0x00)
    }

    override fun getFirmwareFileChunkBytes(
        firmwareBytes: ByteArray,
        packetSize: Int
    ): ByteArray {
        val chunkBytes =
            FirmwareChunkProvider.getInitialFirmwareChunkBytes(firmwareBytes, packetSize)

        val bytes = ByteArray(chunkBytes.size + 3)
        bytes[0] = 0x05
        bytes[1] = 0xAF.toByte()
        val hexCode = Integer.toHexString(chunkBytes.size)
        bytes[2] = hexCode.toInt(16).toByte()

        for (i in chunkBytes.indices) {
            bytes[i + 3] = chunkBytes[i]
        }

        return bytes
    }

    override fun getFirmwareFileChunkBytes(
        firmwareBytes: ByteArray,
        replyBytes: ByteArray,
        packetSize: Int
    ): ByteArray {
        val chunkBytes =
            FirmwareChunkProvider.getFirmwareFileChunkBytes(firmwareBytes, replyBytes, packetSize)

        val bytes = ByteArray(chunkBytes.size + 3)
        bytes[0] = 0x05
        bytes[1] = 0xAF.toByte()
        val hexCode = Integer.toHexString(chunkBytes.size)
        bytes[2] = hexCode.toInt(16).toByte()

        for (i in chunkBytes.indices) {
            bytes[i + 3] = chunkBytes[i]
        }

        return bytes
    }

    override fun getHorizontalSpeedsSecPerRound(version: Int): IntArray {
        val speeds = IntArray(64800)
        for (i in 1..64800) {
            speeds[i - 1] = i
        }
        return speeds
    }

    private var inquiryForAuth = 0
    override fun getAuthenticatePodBytes(
        deviceCategory: PivoDeviceCategory,
        seqNum: Byte,
        answerFromPod: Int,
        inquiryFromPod: Int
    ): ByteArray? {
        val current: ByteArray
        if (seqNum.toInt() == 0) {
            val randInquiry = Random(System.currentTimeMillis())
            val a = randInquiry.nextInt()

            current = byteArrayOf(
                0x05, 0xAF.toByte(),
                0x0D/*length of bytes*/,
                0x41, 0x6F, 0x00, 0x00,
                0x00/*result*/,
                (a shr 24).toByte(),
                (a shr 16).toByte(), (a shr 8).toByte(), a.toByte(), 0x00, 0x00, 0x00, 0x00
            )
            inquiryForAuth = a
        } else if (seqNum.toInt() == 2) {// Error: Pivo Authenticatin Fail!
            val authMode = PodAuthenticator(deviceCategory)
            if (!authMode.verifyAuth(inquiryForAuth, answerFromPod)) {

//                current = byteArrayOf(
//                    0x05, 0xAF.toByte(),
//                    0x0D,/*length of bytes*/
//                    0x41, 0x6F, 0x00, 0x02,
//                    0x00/*result*/,
//                    0x00,
//                    0x00,
//                    0x00,
//                    0x00,
//                    0x00,
//                    0x00,
//                    0x00,
//                    0x00
//                )
                return null
            } else {// Inform: Pivo authentication succeed!
                val code: Int = authMode.getAnswer(inquiryFromPod)

                current = byteArrayOf(
                    0x05, 0xAF.toByte(),
                    0x0D, /* length of bytes */
                    0x41, 0x6F, 0x00/*direction*/, 0x02/*sequence*/,
                    0x01/*result*/,
                    0x00,
                    0x00,
                    0x00,
                    0x00,
                    (code shr 24).toByte(),
                    (code shr 16).toByte(),
                    (code shr 8).toByte(),
                    code.toByte()
                )
            }
        } else {// invalid sequence number
            return null
        }
        return current
    }
}