package app.pivo.android.podsdk.util

import app.pivo.android.podsdk.model.PodModel
import java.util.*

/**
 * Created by murodjon on 2021/12/10
 */
internal object PivoSerialNumberUtil {
    fun getSerialNumber(
        version: Int,
        podModel: PodModel,
        serialNumReplyBytes: ByteArray,
        macAddress: String?
    ): String {
        return if (version == 5) {
            generateSerialNumberForV5Only(version, podModel, macAddress)
        } else if (version >= 6) {
            val serialNumBytes = getSerialNumberBytes(serialNumReplyBytes)
            if (serialNumBytes == null) {
                returnNoSerialNumProvided()
            } else {
                convertBytesToReadableSerialNumber(serialNumBytes)
            }
        } else {
            returnNoSerialNumProvided()
        }
    }

    private fun getSerialNumberBytes(data: ByteArray): ByteArray? {
        return if (isValidSerialNumberBytes(data)) {
            data.copyOfRange(3, 15)
        } else {
            null
        }
    }

    private fun isValidSerialNumberBytes(data: ByteArray): Boolean {
        return data.size == 15
    }

    private fun generateSerialNumberForV5Only(version: Int, podModel: PodModel, macAddress: String?): String {
        if (macAddress == null) return returnNoSerialNumProvided()
        val macAddressNumbers = macAddress.split(":").toTypedArray()
        var code = ""
        if (macAddressNumbers.size == 6) {
            code = macAddressNumbers[3] + macAddressNumbers[4] + macAddressNumbers[5]
        }
        return "P" + getProductName(podModel) + version + "SH" + code
    }

    private fun getProductName(pivoType: PodModel): String? {
        val productNames: MutableMap<PodModel, String> = EnumMap(PodModel::class.java)
        productNames[PodModel.RED] = "10"
        productNames[PodModel.SILVER] = "20"
        productNames[PodModel.BLACK] = "30"
        productNames[PodModel.GOLD] = "40"
        productNames[PodModel.LITE] = "50"
        return productNames[pivoType]
    }

    private fun convertBytesToReadableSerialNumber(data: ByteArray): String {
        val serialNum = StringBuilder()
        for (i in data.indices) {
            if (i <= 2 || i in 4..5) serialNum.append(data[i].toInt().toChar()) else if (i == 3) {
                serialNum.append(data[i])
            } else if (i >= 9) {
                serialNum.append(String.format("%02X", data[i]))
            }
        }
        return serialNum.toString()
    }

    private fun returnNoSerialNumProvided(): String {
        return "No Serial Number Provided"
    }
}