package app.pivo.android.podsdk.cmd.builder.impl

import app.pivo.android.podsdk.auth.PodAuthenticator
import app.pivo.android.podsdk.cmd.builder.CommandBuilder
import app.pivo.android.podsdk.cmd.builder.ext.getHexValue
import app.pivo.android.podsdk.model.*
import app.pivo.android.podsdk.ota.FirmwareChunkProvider
import app.pivo.android.podsdk.util.HexUtil.convertHorizontalDegreeToHexByte
import app.pivo.android.podsdk.util.HexUtil.convertSpeedToHexBytes
import app.pivo.android.podsdk.util.HexUtil.convertVerticalDegreeToHexByte
import java.util.*

/**
 * Created by murodjon on 2021/05/21
 */
internal class PodXCmdBuilderImpl : CommandBuilder {

    /**
     * Get OTA initiation bytes
     */
    override fun getOTAInitiationBytes(): ByteArray {
        return byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x55, 0x8F.toByte(), 0x01, 0x00, 0x00)
    }

    override fun getFirmwareFileChunkBytes(firmwareBytes: ByteArray, packetSize: Int): ByteArray {
        val chunkBytes =
            FirmwareChunkProvider.getInitialFirmwareChunkBytes(firmwareBytes, packetSize)

        val bytes = ByteArray(chunkBytes.size + 3)
        bytes[0] = 0x05
        bytes[1] = 0xAF.toByte()
        val hexCode = Integer.toHexString(chunkBytes.size)
        bytes[2] = hexCode.toInt(16).toByte()

        for (i in chunkBytes.indices) {
            bytes[i + 3] = chunkBytes[i]
        }
        return bytes
    }

    override fun getFirmwareFileChunkBytes(
        firmwareBytes: ByteArray,
        replyBytes: ByteArray,
        packetSize: Int
    ): ByteArray {
        val chunkBytes =
            FirmwareChunkProvider.getFirmwareFileChunkBytes(firmwareBytes, replyBytes, packetSize)

        val bytes = ByteArray(chunkBytes.size + 3)
        bytes[0] = 0x05
        bytes[1] = 0xAF.toByte()
        val hexCode = Integer.toHexString(chunkBytes.size)
        bytes[2] = hexCode.toInt(16).toByte()

        for (i in chunkBytes.indices) {
            bytes[i + 3] = chunkBytes[i]
        }

        return bytes
    }

    /**
     * get check connection bytes
     */
    override fun getDeviceInfoBytes() =
        byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x42, 0x01, 0x01, 0x00, 0x00)

    /**
     * This function returns timeout bytes
     */
    override fun getTimeoutBytes(time: Int): ByteArray {
        return if (time % 10 == 0 && time >= 0 && time <= 2550) {
            byteArrayOf(
                0x05,
                0xAF.toByte(),
                0x05,
                0x42.toByte(),
                0x02,
                0x01,
                (time / 10).toByte(),
                0x00
            )
        } else {
            byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x42.toByte(), 0x02, 0x01, 0x00, 0x00)
        }
    }

    /**
     * get battery level bytes
     */
    override fun getBatteryLevelBytes() =
        byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x42, 0x03, 0x01, 0x00, 0x00)

    /**
     * This function returns sound command bytes according to sound type
     * @param sound is predefined sound type
     */
    override fun getSoundBytes(sound: SoundType) =
        byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x42, 0x04, 0x01, sound.getHexValue(), 0x00)

    /**
     * get mac address requester bytes
     * @return an array of byte values
     */
    override fun getMacAddressBytes(): ByteArray =
        byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x49, 0x1B, 0x01, 0x00, 0x00)

    /**
     * get serial number requester bytes
     */
    override fun getSerialNumberBytes(): ByteArray =
        byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x49, 0x1C, 0x01, 0x00, 0x00)


    /**
     * get change name bytes
     * @param name is new name for headset
     * @return an array of byte values is returned from function
     */
    override fun getChangeNameBytes(name: String): ByteArray? {
        val length = name.length
        if (length == 0) return null
        // final byte array size for changing name is 5+(length of name)
        val bytes = ByteArray(length + 6)
        bytes[0] = 0x05.toByte()
        bytes[1] = 0xAF.toByte()
        bytes[2] = (length + 3).toByte()
        bytes[3] = 0x49.toByte()
        bytes[4] = 0x7F.toByte()
        bytes[5] = 0x01.toByte()
        name.forEachIndexed { i, v ->
            bytes[6 + i] = name[i].code.toByte()
        }
        return bytes
    }

    /**
     * This function returns remote controller switcher bytes according to
     * @param remoteEnabled is true when remote controller and pod communicates
     */
    override fun getRemoteSwitcherBytes(remoteEnabled: Boolean) =
        byteArrayOf(
            0x05,
            0xAF.toByte(),
            0x05,
            0x52,
            0x01,
            0x01,
            if (remoteEnabled) 0xA1.toByte() else 0xA0.toByte(),
            0x00
        )

    /**
     * get start pairing bytes
     */
    override fun getStartPairingBytes() =
        byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x52, 0x02, 0x01, 0x00, 0x00)

    /**
     * get cancel pairing bytes
     */
    override fun getCancelPairingBytes() =
        byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x52, 0x03, 0x01, 0x00, 0x00)


    /**
     * This function is used to enable and disable notifier according
     * @param notifierEnabled notifier is enabled when notifierEnabled true
     */
    override fun getNotifierSwitcherBytes(notifierEnabled: Boolean) =
        byteArrayOf(
            0x05,
            0xAF.toByte(),
            0x05,
            0x4D,
            0xF1.toByte(),
            0x01,
            if (notifierEnabled) 0xA1.toByte() else 0xA0.toByte(),
            0x00
        )

    /**
     * *********************************************************************************
     * *********************************************************************************
     * *****************<<<< Movement Control Command Set >>>>**************************
     * *********************************************************************************
     * *********************************************************************************
     */

    /**
     * get battery level bytes
     */
    override fun getStopBytes() =
        byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x4D, 0x00, 0x01, 0x00, 0x00)

    override fun getHorizontalSpeedsSecPerRound(version: Int): IntArray {
        val speeds = IntArray(64800)
        for (i in 1..64800) {
            speeds[i - 1] = i
        }
        return speeds
    }

    /**
     * get horizontal movement bytes
     * @param speed is speed for horizontal
     * @param degree is rotation degree
     * @param direction is direction of movement
     */
    override fun getMoveHorizontallyBytes(
        version: Int,
        direction: HorizontalDirection,
        speed: Speed,
        degree: Int
    ): ByteArray? {
        val directionByte: Byte = if (direction == HorizontalDirection.LEFT)
            if (speed.speedUnit == SpeedUnit.SEC_PER_ROUND) 0x21 else 0x61
        else if (speed.speedUnit == SpeedUnit.SEC_PER_ROUND) 0x20 else 0x60

        val speedBytes = convertSpeedToHexBytes(speed.value) ?: return null
        val rotationBytes = convertHorizontalDegreeToHexByte(degree) ?: return null

        val data = byteArrayOf(
            0x05, 0xAF.toByte(), 0x07, 0x4D, /* direction */directionByte, 0x01,
            /* speed1 */speedBytes[0], /* speed1 */speedBytes[1],
            /* degree1 */rotationBytes[0], /* degree1 */rotationBytes[1]
        )
        return data
    }

    override fun getHoldTurningHorizontallyBytes(
        version: Int,
        direction: HorizontalDirection,
        speed: Speed
    ): ByteArray? {
        val directionByte: Byte = if (direction == HorizontalDirection.LEFT)
            if (speed.speedUnit == SpeedUnit.SEC_PER_ROUND) 0x21 else 0x61
        else if (speed.speedUnit == SpeedUnit.SEC_PER_ROUND) 0x20 else 0x60

        val speedBytes = convertSpeedToHexBytes(speed.value) ?: return null
        val rotationBytes = convertHorizontalDegreeToHexByte(1) ?: return null

        val data = byteArrayOf(
            0x05, 0xAF.toByte(), 0x07, 0x4D, /* direction */directionByte, 0x01,
            /* speed1 */speedBytes[0], /* speed1 */speedBytes[1],
            /* degree1 */rotationBytes[0], /* degree1 */rotationBytes[1]
        )
        return data
    }

    /**
     * This returns bytes for getting horizontal position bytes
     */
    override fun getHorizontalPositionBytes() =
        byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x4D, 0x22, 0x01, 0x00, 0x00)

    /**
     * This returns "go to horizontal position" bytes according to speed and degree
     */
    override fun getGoToHorizontalPositionBytes(speed: Speed, degree: Int): ByteArray? {
        val speedBytes = convertSpeedToHexBytes(speed.value) ?: return null
        val rotationBytes = convertHorizontalDegreeToHexByte(degree) ?: return null
        val speedUnitByte: Byte = if (speed.speedUnit == SpeedUnit.SEC_PER_ROUND) 0x23 else 0x63

        return byteArrayOf(
            0x05, 0xAF.toByte(), 0x07, 0x4D, speedUnitByte, 0x01,
            /* speed1 */speedBytes[0], /* speed1 */speedBytes[1],
            /* degree1 */rotationBytes[0], /* degree1 */rotationBytes[1]
        )
    }

    /**
     * This returns bytes to set speed for horizontal motor
     * @param speed
     */
    override fun getSetHorizontalSpeedBytes(version: Int, speed: Speed): ByteArray? {
        val speedBytes = convertSpeedToHexBytes(speed.value) ?: return null
        val speedUnitByte: Byte = if (speed.speedUnit == SpeedUnit.SEC_PER_ROUND) 0x24 else 0x64

        return byteArrayOf(
            0x05, 0xAF.toByte(), 0x05, 0x4D, speedUnitByte, 0x01,
            /* speed1 */speedBytes[0], /* speed1 */speedBytes[1]
        )
    }

    override fun getResetPositionBytes() =
        byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x4D, 0x29, 0x01, 0x00, 0x00)

    /**
     * get vertical movement bytes
     * @param speed is speed for vertical movement
     * @param degree is rotation degree
     * @param direction is moving direction
     */
    override fun getMoveVerticallyBytes(
        direction: VerticalDirection,
        speed: Speed,
        degree: Int
    ): ByteArray? {
        val directionByte: Byte = if (direction == VerticalDirection.UP) 0x30 else 0x31
        val speedBytes = convertSpeedToHexBytes(speed.value) ?: return null
        val rotationByte = convertVerticalDegreeToHexByte(degree) ?: return null
        val data = byteArrayOf(
            0x05, 0xAF.toByte(), 0x06, 0x4D,/* direction */directionByte, 0x01,
            /* speed1 */speedBytes[0], /* speed1 */speedBytes[1],
            /* degree1 */rotationByte
        )
        return data
    }

    /**
     * This returns bytes for getting vertical position bytes
     */
    override fun getVerticalPositionBytes() =
        byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x4D, 0x32, 0x01, 0x00, 0x00)

    /**
     * This returns "go to vertical position" bytes according to speed and degree
     */
    override fun getGoToVerticalPositionBytes(speed: Speed, degree: Int): ByteArray? {
        val speedBytes = convertSpeedToHexBytes(speed.value) ?: return null
        val rotationByte = convertVerticalDegreeToHexByte(degree) ?: return null
        val speedUnitByte:Byte =
        return byteArrayOf(
            0x05, 0xAF.toByte(), 0x06, 0x4D, 0x33, 0x01,
            /* speed1 */speedBytes[0], /* speed1 */speedBytes[1],
            /* degree1 */rotationByte
        )
    }

    /**
     * This returns bytes to set speed for vertical motor
     * @param speed
     */
    override fun getSetVerticalSpeedBytes(speed: Speed): ByteArray? {
        val speedBytes = convertSpeedToHexBytes(speed.value) ?: return null
        return byteArrayOf(
            0x05, 0xAF.toByte(), 0x05, 0x4D, 0x34, 0x01,
            /* speed1 */speedBytes[0], /* speed1 */speedBytes[1]
        )
    }

    /**
     * get Moving combination bytes
     *
     * @param hDirection is horizontal movement direction
     * @param hSpeed is horizontal movement speed
     * @param hDegree is horizontal movement degree
     *
     * @param vlDirection is vertical movement direction
     * @param vSpeed is vertical movement speed
     * @param vDegree is vertical movement rotation degree
     */
    override fun getMovingCombinationBytes(
        hSpeed: Speed,
        hDegree: Int,
        vSpeed: Speed,
        vDegree: Int,
        mixedDirection: MixedDirection
    ): ByteArray? {

        val hSpeeds = convertSpeedToHexBytes(hSpeed.value) ?: return null
        val vSpeeds = convertSpeedToHexBytes(vSpeed.value) ?: return null

        val hRotationBytes = convertHorizontalDegreeToHexByte(hDegree) ?: return null
        val vRotationByte = convertVerticalDegreeToHexByte(vDegree) ?: return null

        val directionByte: Byte = when (mixedDirection) {
            MixedDirection.RIGHT_DOWN -> 0x42
            MixedDirection.RIGHT_UP -> 0x41
            MixedDirection.LEFT_DOWN -> 0x44
            else -> 0x43
        }

        val data = byteArrayOf(
            0x05, 0xAF.toByte(), 0xA, 0x4D, /* direction */directionByte, 0x001,
            /* speed1 */hSpeeds[0], /* speed1 */hSpeeds[1],
            /* degree1 */hRotationBytes[0], /* degree1 */hRotationBytes[1],
            /* speed1 */vSpeeds[0], /* speed1 */vSpeeds[1],
            vRotationByte
        )
        return data
    }

    /**
     * This returns bytes to set LED color based on
     * @param colors
     */
    override fun getSetLEDColorBytes(colors: List<LEDColor>, brightness: Int): ByteArray? {
        if (colors.size != 12 && brightness !in 0..100) return null
        val data = byteArrayOf(
            0x05,
            0xAF.toByte(),
            0xF,
            0x4C,
            0x40,
            0x01,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            0x00,
            brightness.toByte()
        )
        colors.forEachIndexed { index, ledColor -> data[index + 6] = ledColor.getValue() }
        return data
    }

    /**
     * This returns bytes to retrieve current set LED color
     */
    override fun getRetrieveLEDColorBytes() =
        byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x4C, 0x41, 0x01, 0x00, 0x00)

    override fun getRunLightPresetBytes(patternType: LightPatternType, level: Int): ByteArray? {
        val patternByte: Byte = when (patternType) {
            LightPatternType.PatternA -> 0x01
            LightPatternType.PatternB -> 0x02
            LightPatternType.PatternC -> 0x03
            LightPatternType.PatternD -> 0x04
            LightPatternType.PatternE -> 0x05
            LightPatternType.PatternF -> 0x06
            LightPatternType.PatternG -> 0x07
            LightPatternType.PatternH -> 0x08
        }
        return byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x4C, 0x21, 0x01, patternByte, level.toByte())
    }

    override fun getStopLightPresetBytes() =
        byteArrayOf(0x05, 0xAF.toByte(), 0x05, 0x4C, 0x20, 0x01, 0x00, 0x00)

    private var inquiryForAuth = 0
    override fun getAuthenticatePodBytes(
        deviceCategory: PivoDeviceCategory,
        seqNum: Byte,
        answerFromPod: Int,
        inquiryFromPod: Int
    ): ByteArray? {
        val current: ByteArray
        if (seqNum.toInt() == 0) {
            val randInquiry = Random(System.currentTimeMillis())
            val a = randInquiry.nextInt()

            current = byteArrayOf(
                0x05, 0xAF.toByte(),
                0x0D/*length of bytes*/,
                0x41, 0x6F, 0x00, 0x00,
                0x00/*result*/,
                (a shr 24).toByte(),
                (a shr 16).toByte(), (a shr 8).toByte(), a.toByte(), 0x00, 0x00, 0x00, 0x00
            )
            inquiryForAuth = a
        } else if (seqNum.toInt() == 2) {// Error: Pivo Authenticatin Fail!
            val authMode = PodAuthenticator(deviceCategory)
            if (!authMode.verifyAuth(inquiryForAuth, answerFromPod)) {

//                current = byteArrayOf(
//                    0x05, 0xAF.toByte(),
//                    0x0D,/*length of bytes*/
//                    0x41, 0x6F, 0x00, 0x02,
//                    0x00/*result*/,
//                    0x00,
//                    0x00,
//                    0x00,
//                    0x00,
//                    0x00,
//                    0x00,
//                    0x00,
//                    0x00
//                )
                return null
            } else {// Inform: Pivo authentication succeed!
                val code: Int = authMode.getAnswer(inquiryFromPod)

                current = byteArrayOf(
                    0x05, 0xAF.toByte(),
                    0x0D, /* length of bytes */
                    0x41, 0x6F, 0x00/*direction*/, 0x02/*sequence*/,
                    0x01/*result*/,
                    0x00,
                    0x00,
                    0x00,
                    0x00,
                    (code shr 24).toByte(),
                    (code shr 16).toByte(),
                    (code shr 8).toByte(),
                    code.toByte()
                )
            }
        } else {// invalid sequence number
            return null
        }
        return current
    }
}