package app.pivo.android.podsdk.cmd.parser.impl

import app.pivo.android.podsdk.cmd.parser.impl.pod.*
import app.pivo.android.podsdk.cmd.parser.impl.pod.ConnectionTimeoutReply
import app.pivo.android.podsdk.cmd.parser.impl.pod.NotifierReply
import app.pivo.android.podsdk.cmd.parser.impl.pod.PairingReply
import app.pivo.android.podsdk.cmd.parser.impl.pod.StopReply
import app.pivo.android.podsdk.cmd.parser.ReplyCommandParser
import app.pivo.android.podsdk.cmd.parser.impl.pod.AuthReply
import app.pivo.android.podsdk.cmd.parser.impl.pod.BatteryReply
import app.pivo.android.podsdk.cmd.parser.impl.pod.DeviceInfoReply
import app.pivo.android.podsdk.cmd.parser.impl.pod.RemoteConEnabledReply
import app.pivo.android.podsdk.cmd.parser.impl.pod.SoundReply
import app.pivo.android.podsdk.model.*
import app.pivo.android.podsdk.model.PivoEvent
import app.pivo.android.podsdk.util.PivoSerialNumberUtil

/**
 * Created by murodjon on 2022/01/13
 */
internal class PodReplyCommandParserImpl(private val podDevice: PivoDevice) : ReplyCommandParser {

    private var deviceInfo: DeviceInfo? = null

    override fun getEvent(bytes: ByteArray): PivoEvent {
        return getType(bytes)
    }

    private fun getType(bytes: ByteArray): PivoEvent {
        return when {
            // basic command set
            DeviceInfoReply.isDeviceInfoReply(bytes) -> {
                val deviceInfo = DeviceInfoReply(bytes).getData()
                this.deviceInfo = deviceInfo.deviceInfo
                deviceInfo
            }
            AuthReply.isAuthNotification(bytes) -> PivoEvent.AuthenticationEvent(AuthReply(bytes).getData())
            ConnectionTimeoutReply.isTimeoutReply(bytes) -> PivoEvent.ConnectionTimeOutEvent(
                ConnectionTimeoutReply(bytes).getData()
            )
            BatteryReply.isBatteryReply(bytes) -> PivoEvent.BatteryEvent(BatteryReply(bytes).getData())
            MacAddressReply.isMacAddressReply(bytes) -> {
                val macAddress = MacAddressReply(bytes).getData()
                if (macAddress == null) {
                    PivoEvent.InvalidReplyEvent(bytes)
                } else {
                    PivoEvent.MacAddressEvent(macAddress)
                }
            }
            SerialNumberReply.isSerialNumberReply(bytes) -> {
                var podModel: PodModel? = null

                if(deviceInfo != null && deviceInfo!!.getPivoCategory() is PivoDeviceCategory.PivoPod) {
                    podModel = (deviceInfo!!.getPivoCategory() as PivoDeviceCategory.PivoPod?)?.model
                }

                PivoEvent.SerialNumberEvent(
                    SerialNumber(
                        PivoSerialNumberUtil.getSerialNumber(
                            version = deviceInfo?.getPivoCategory()?.version ?: 0,
                            podModel = podModel ?: PodModel.BLACK,
                            serialNumReplyBytes = bytes,
                            podDevice.getMacAddress()
                        )
                    )
                )
            }

            // movement control command set
            StopReply.isStopReply(bytes) -> PivoEvent.Stop
            // remove controller command set
            RemoteConEnabledReply.isRemoteBypassReply(bytes) -> PivoEvent.RemoteConBypassEvent(
                RemoteConEnabledReply(bytes).getData()
            )
            PairingReply.isPairing(bytes) -> PivoEvent.RemoteConPairingEvent(PairingReply(bytes).getData())

            // Change device name
            NameChangedReply.isNameChangedReply(bytes) -> {
                val name = NameChangedReply(bytes).getData()
                if (name == null) {
                    PivoEvent.InvalidReplyEvent(bytes)
                } else {
                    PivoEvent.NameChangedEvent(name)
                }
            }
            SoundReply.isSoundReply(bytes) -> {
                val sound = SoundReply(bytes).getData()
                if (sound == null) {
                    PivoEvent.InvalidReplyEvent(bytes)
                } else {
                    PivoEvent.SoundEvent(sound)
                }
            }
            NotifierReply.isNotifierReply(bytes) -> PivoEvent.NotifierEvent(NotifierReply(bytes).getData())
            RemoteControllerReply.isRemoteConPressedReply(bytes) -> {
                val remoteCon = RemoteControllerReply(bytes).getData()
                if (remoteCon == null) {
                    PivoEvent.InvalidReplyEvent(bytes)
                } else {
                    PivoEvent.RemoteConButtonEvent(remoteCon)
                }
            }
            HorizontalMovementReply.isHorizontalMovementReply(bytes) -> {
                val movement = HorizontalMovementReply(
                    bytes
                ).getData()
                if (movement == null) {
                    PivoEvent.InvalidReplyEvent(bytes)
                } else {
                    PivoEvent.HorizontalMovementEvent(movement)
                }
            }
            HorizontalSpeedReply.isHorizontalSpeedReply(bytes) -> {
                val speed = HorizontalSpeedReply(bytes).getData()
                if (speed == null) {
                    PivoEvent.InvalidReplyEvent(bytes)
                } else {
                    PivoEvent.SpeedEvent(speed)
                }
            }
            else -> PivoEvent.InvalidReplyEvent(bytes)
        }
    }
}