package app.pivo.android.prosdk;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import app.pivo.android.prosdk.util.PivoType;
import app.pivo.android.prosdk.util.Version;

/**
 * Created by murodjon on 2021/12/10
 */
class PivoSerialNumberUtil {

    static String getSerialNumber(Version version, byte[] serialNumReplyBytes, String macAddress) {
        if (version.getVersion() == 5) {
            return generateSerialNumberForV5Only(version, macAddress);
        } else if (version.getVersion() >= 6) {
            byte[] serialNumBytes = getSerialNumberBytes(serialNumReplyBytes);
            if (serialNumBytes == null) {
                return returnNoSerialNumProvided();
            } else {
                return convertBytesToReadableSerialNumber(serialNumBytes);
            }
        } else {
            return returnNoSerialNumProvided();
        }
    }

    private static byte[] getSerialNumberBytes(byte[] data) {
        if (isValidSerialNumberBytes(data)) {
            return Arrays.copyOfRange(data, 3, 15);
        } else {
            return null;
        }
    }

    private static boolean isValidSerialNumberBytes(byte[] data) {
        return data.length == 15;
    }

    private static String generateSerialNumberForV5Only(Version version, String macAddress) {
        if (macAddress == null) return returnNoSerialNumProvided();
        String[] macAddressNumbers = macAddress.split(":");
        String code = "";
        if (macAddressNumbers.length == 6) {
            code = macAddressNumbers[3] + macAddressNumbers[4] + macAddressNumbers[5];
        }
        return "P" + getProductName(version.getPivoType()) + version.getVersion() + "SH" + code;
    }

    private static String getProductName(PivoType pivoType) {
        Map<PivoType, String> productNames = new HashMap<>();
        productNames.put(PivoType.RED, "10");
        productNames.put(PivoType.SILVER, "20");
        productNames.put(PivoType.BLACK, "30");
        productNames.put(PivoType.GOLD, "40");
        productNames.put(PivoType.LITE, "50");
        return productNames.get(pivoType);
    }

    private static String convertBytesToReadableSerialNumber(byte[] data) {
        StringBuilder serialNum = new StringBuilder();
        for (int i = 0; i < data.length; i++) {
            if (i <= 2 || (i >= 4 && i <= 5))
                serialNum.append((char) data[i]);
            else if (i == 3) {
                serialNum.append(data[i]);
            } else if (i >= 9) {
                serialNum.append(String.format("%02X", data[i]));
            }
        }
        return serialNum.toString();
    }

    private static String returnNoSerialNumProvided() {
        return "No Serial Number Provided";
    }
}
