package app.pivo.android.prosdk.events;

import app.pivo.android.prosdk.util.Direction;
import app.pivo.android.prosdk.util.PivoDevice;
import app.pivo.android.prosdk.util.PivoDeviceInfo;

/**
 * PivoEvent is a list of events, which are sent by Pivo controller through @{@link PivoEventBus} mechanism.
 */
public class PivoEvent {

    public static int UP = 0;
    public static int DOWN = 1;

    public static int PRESSED = 0;
    public static int RELEASE = 1;

    /**
     * ConnectionFailure event is published if Pivo disconnects from an application.
     */
    public static class ConnectionFailure {
    }

    /**
     * {@link ConnectionComplete} event is sent if {@link app.pivo.android.prosdk.PivoProSdk#connectTo(PivoDevice)} completes the connection to Pivo device.
     */
    public static class ConnectionComplete {
        private final PivoDeviceInfo pivoDeviceInfo;

        public ConnectionComplete(PivoDeviceInfo pivoDeviceInfo) {
            this.pivoDeviceInfo = pivoDeviceInfo;
        }

        public PivoDeviceInfo getPivoDeviceInfo() {
            return pivoDeviceInfo;
        }
    }

    /**
     * {@link Scanning} is sent by {@link PivoEventBus} with found Pivo devices while pivo scanning process.
     */
    public static class Scanning {
        PivoDevice device;

        public Scanning(PivoDevice device) {
            this.device = device;
        }

        public PivoDevice getDevice() {
            return device;
        }
    }

    /**
     * {@link Version} is a version of connected Pivo
     */
    public static class Version {
        int version;

        public Version(int version) {
            this.version = version;
        }

        public int getVersion() {
            return version;
        }
    }

    /**
     * {@link BatteryChanged} is sent by sdk to inform the app about remaining battery level.
     */
    public static class BatteryChanged {
        int level;

        public BatteryChanged(int level) {
            this.level = level;
        }

        public int getLevel() {
            return level;
        }
    }

    /**
     * {@link Timeout} is sent when app sends connection timeout.
     */
    public static class Timeout {
        int time;

        public Timeout(int time) {
            this.time = time;
        }

        public int getTime() {
            return time;
        }
    }

    /**
     * {@link MacAddress } is sent by {@link PivoEventBus} after receiving Mac address from pivo.
     */
    public static class MacAddress {
        String name;

        public MacAddress(String name) {
            this.name = name;
        }

        public String getMacAddress() {
            return name;
        }
    }

    /**
     * {@link NameChanged } is sent by {@link PivoEventBus} after Pivo name is changed successfully.
     */
    public static class NameChanged {
        String name;

        public NameChanged(String name) {
            this.name = name;
        }

        public String getName() {
            return name;
        }
    }

    public static class Off {
    }

    /**
     * {@link Rotated} is sent by {@link PivoEventBus} to notify app about rotation.
     */
    public static class Rotated {
        private final Direction direction;

        public Rotated(Direction direction) {
            this.direction = direction;
        }

        public Direction getDirection() {
            return direction;
        }
    }

    /**
     * {@link RCMode} is sent by {@link PivoEventBus} if the Pivo remote controller mode pressed.
     */
    public static class RCMode {
        private int state = 0;

        public RCMode() {
        }

        public RCMode(int state) {
            this.state = state;
        }

        public int getState() {
            return state;
        }
    }

    /**
     * {@link RCCamera} is sent by {@link PivoEventBus} if camera button is pressed on Pivo remote controller.
     */
    public static class RCCamera {
        private int state = 0;

        public RCCamera() {
        }

        public RCCamera(int state) {
            this.state = state;
        }

        public int getState() {
            return state;
        }
    }

    /**
     * {@link RCStop} is sent by {@link PivoEventBus} if stop button is pressed on Pivo remote controller.
     */
    public static class RCStop {
        private int state = 0;

        public RCStop() {
        }

        public RCStop(int state) {
            this.state = state;
        }

        public int getState() {
            return state;
        }
    }

    /**
     * {@link RCLeft} is sent by {@link PivoEventBus} if left button is pressed on Pivo remote controller.
     */
    public static class RCLeft {
        private int state = 0;

        public RCLeft() {
        }

        public RCLeft(int state) {
            this.state = state;
        }

        public int getState() {
            return state;
        }
    }


    /**
     * This event class is deprecated and please use {@link RCLeft} instead of it.
     * {@link RCLeftPressed} is sent by {@link PivoEventBus} if left button is pressed on Pivo remote controller.
     */
    @Deprecated
    public static class RCLeftPressed {
    }

    /**
     * {@link RCRight} is sent by {@link PivoEventBus} if left button is pressed on Pivo remote controller.
     */
    public static class RCRight {
        private int state = 0;

        public RCRight() {
        }

        public RCRight(int state) {
            this.state = state;
        }

        public int getState() {
            return state;
        }
    }

    /**
     * This event class is deprecated and please use {@link RCRight} instead of it.
     * {@link RCRightPressed} is sent by {@link PivoEventBus} if right button is pressed on Pivo remote controller
     */
    @Deprecated
    public static class RCRightPressed {
    }

    /**
     * This event class is deprecated and please use {@link RCRight} instead of it.
     * {@link RCRightRelease} is sent by {@link PivoEventBus} if right button is released after pressing the button on Pivo remote controller.
     */
    @Deprecated
    public static class RCRightRelease {
    }

    /**
     * This event class is deprecated and please use {@link RCLeft} instead of it.
     * {@link RCLeftRelease} is sent by {@link PivoEventBus} if left button is released after pressing the button on Pivo remote controller.
     */
    @Deprecated
    public static class RCLeftRelease {
    }

    /**
     * {@link RCLeftContinuous} is sent by {@link PivoEventBus} if left button is released after pressing the button on Pivo remote controller.
     */
    public static class RCLeftContinuous {
        private int state = 0;

        private RCLeftContinuous() {
        }

        public RCLeftContinuous(int state) {
            this.state = state;
        }

        public int getState() {
            return state;
        }
    }

    /**
     * {@link RCRightContinuous} is sent by {@link PivoEventBus} if right button is released after pressing the button on Pivo remote controller.
     */
    public static class RCRightContinuous {
        private int state = 0;

        private RCRightContinuous() {
        }

        public RCRightContinuous(int state) {
            this.state = state;
        }

        public int getState() {
            return state;
        }
    }

    /**
     * {@link RCSpeed} is sent by {@link PivoEventBus} if speed up and down buttons pressed
     */
    public static class RCSpeed {
        int level;
        int buttonType = -1;
        int state = 0;

        public RCSpeed(int speedLevel) {
            this.level = speedLevel;
        }

        public RCSpeed(int speedLevel, int state) {
            this.level = speedLevel;
            this.state = state;
        }

        public RCSpeed(int speedLevel, int buttonType, int state) {
            this.level = speedLevel;
            this.buttonType = buttonType;
            this.state = state;
        }

        public int getLevel() {
            return level;
        }

        public int getButtonType() {
            return buttonType;
        }

        public int getState() {
            return state;
        }
    }

    /**
     * {@link RCSpeedUp} is sent by {@link PivoEventBus} on pressing up speed button.
     * This event is supported in Pivo version-1 and above.
     */
    public static class RCSpeedUp {
        private final int level;
        private int state = 0;

        public RCSpeedUp(int level, int state) {
            this.level = level;
            this.state = state;
        }

        public int getLevel() {
            return level;
        }

        public int getState() {
            return state;
        }
    }

    /**
     * {@link BypassEvent} is sent by {@link PivoEventBus} when remote controller is activated,
     * or in other words, bypass is turned off. This is supported in Pivo version-2 and above.
     */
    public static class BypassEvent {
        boolean enabled;

        public BypassEvent(boolean enabled) {
            this.enabled = enabled;
        }

        public boolean isEnabled() {
            return enabled;
        }
    }

    /**
     * This event class is deprecated and please use {@link RCSpeed} instead of it.
     * {@link RCSpeedUpPressed} is sent by {@link PivoEventBus} on pressing up speed button.
     * This event is supported in Pivo version-1 and above.
     */
    @Deprecated
    public static class RCSpeedUpPressed {
        private final int level;

        public RCSpeedUpPressed(int level) {
            this.level = level;
        }

        public int getLevel() {
            return level;
        }
    }

    /**
     * This event class is deprecated and please use {@link RCSpeed} instead of it.
     * {@link RCSpeedUpRelease} is sent by {@link PivoEventBus} after release up speed button.
     * This is supported in Pivo version-1 and above.
     */
    @Deprecated
    public static class RCSpeedUpRelease {
        private final int level;

        public RCSpeedUpRelease(int level) {
            this.level = level;
        }

        public int getLevel() {
            return level;
        }
    }

    /**
     * This event class is deprecated and please use {@link RCSpeed} instead of it.
     * {@link RCSpeedDown} is sent by {@link PivoEventBus} on pressing up speed button.
     * This event is supported in Pivo version-1 and above.
     */
    public static class RCSpeedDown {
        private final int level;
        private int state = 0;

        public RCSpeedDown(int level, int state) {
            this.level = level;
            this.state = state;
        }

        public int getLevel() {
            return level;
        }

        public int getState() {
            return state;
        }
    }

    /**
     * This event class is deprecated and please use {@link RCSpeed} instead of it.
     * {@link RCSpeedDownPressed} is sent by {@link PivoEventBus} after release up speed button.
     * This is supported in Pivo version-1 and above.
     */
    @Deprecated
    public static class RCSpeedDownPressed {
        private final int level;

        public RCSpeedDownPressed(int level) {
            this.level = level;
        }

        public int getLevel() {
            return level;
        }
    }

    /**
     * This event class is deprecated and please use {@link RCSpeed} instead of it.
     * {@link RCSpeedDownRelease} is sent by {@link PivoEventBus} after release up speed button.
     * This is supported in Pivo version-1 and above.
     */
    @Deprecated
    public static class RCSpeedDownRelease {
        private final int level;

        public RCSpeedDownRelease(int level) {
            this.level = level;
        }

        public int getLevel() {
            return level;
        }
    }
}
