package app.valuationcontrol.multimodule.library.helpers;

import app.valuationcontrol.multimodule.library.entities.Model;
import app.valuationcontrol.multimodule.library.entities.Variable;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Locale;

public abstract class FormulaEvaluator {

  private static final String WORD_BOUNDARY = "\\b";
  public static final String ZERO_OR_MORE_CHARACTERS = ".*";
  private static final String LEFT_CURLY = "{";
  private static final String RIGHT_CURLY = "}";
  private static final Comparator<Variable> VARIABLE_NAME_LENGTH =
      (variable1, variable2) ->
          variable2.getVariableName().length() - variable1.getVariableName().length();
  public static final String PERCENTAGE = "_percentage_";
  public static final String PERCENTAGE_SIGN = "%";

  private FormulaEvaluator() {}

  public static String evaluateVariableFormula(Variable variable) {
    String currentEvaluatedFormula = variable.getVariableFormula().toLowerCase();
    currentEvaluatedFormula = currentEvaluatedFormula.trim(); // remove any blank characters
    // Clear dependencies
    variable.getVariableDependencies().clear();

    // Trying to parse percentages
    try {
      MyCustomFloatEditor myCustomFloatEditor = new MyCustomFloatEditor(Float.class, false);
      myCustomFloatEditor.setAsText(currentEvaluatedFormula);
      return currentEvaluatedFormula;
    } catch (NumberFormatException ignored) {
    }

    try {
      Double.parseDouble(currentEvaluatedFormula);
      return currentEvaluatedFormula;
    } catch (NumberFormatException ignored) {
    }

    try {
      Integer.parseInt(currentEvaluatedFormula);
      return currentEvaluatedFormula;
    } catch (NumberFormatException ignored) {
    }

    Model model = variable.getAttachedModel();

    // get clean formula
    currentEvaluatedFormula = getCleanName(currentEvaluatedFormula);

    // Scan for variables and their position
    // Create a local copy to avoid different sorting throughout the process
    List<Variable> myVariables = new ArrayList<>(model.getVariables());

    myVariables.sort(VARIABLE_NAME_LENGTH);

    for (Variable searchVariable : myVariables) {
      String cleanVariableName =
          getCleanName(searchVariable.getVariableName().toLowerCase(Locale.ROOT));
      String matchStringClean = exactWord(cleanVariableName);
      String replacement = LEFT_CURLY + searchVariable.getId() + RIGHT_CURLY;

      currentEvaluatedFormula = currentEvaluatedFormula.replaceAll(matchStringClean, replacement);

      if (currentEvaluatedFormula.contains(replacement)) {
        variable.getVariableDependencies().add(searchVariable);
      }
    }

    return currentEvaluatedFormula.replace(PERCENTAGE, PERCENTAGE_SIGN);
  }

  private static String getCleanName(String name) {
    return name.replace(PERCENTAGE_SIGN, PERCENTAGE).replace("'", "").toLowerCase();
  }


  /**
   * Produce a regex replacing the exact word
   *
   * @param variableName the word
   * @return regex for the word sent in.
   */
  public static String exactWord(String variableName) {
    return WORD_BOUNDARY + variableName + WORD_BOUNDARY;
  }

  /** produce a regex producing a contains regexp */
  public static String contains(String value) {
    return ZERO_OR_MORE_CHARACTERS + value + ZERO_OR_MORE_CHARACTERS;
  }


}
