package app.valuationcontrol.multimodule.library.powerpoint;

import app.valuationcontrol.multimodule.library.entities.*;
import app.valuationcontrol.multimodule.library.helpers.openai.OpenAIHelperFunctions;
import app.valuationcontrol.multimodule.library.helpers.exceptions.ResourceException;
import app.valuationcontrol.multimodule.library.helpers.openai.OpenAiServiceImplementation;
import app.valuationcontrol.multimodule.library.records.CalculationData;
import app.valuationcontrol.multimodule.library.records.ScenarioComparison;
import app.valuationcontrol.multimodule.library.records.SensitivityData;
import app.valuationcontrol.multimodule.library.records.VariableData;
import app.valuationcontrol.multimodule.library.xlhandler.SCENARIO;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.theokanning.openai.completion.chat.ChatMessage;
import com.theokanning.openai.completion.chat.ChatMessageRole;
import java.awt.*;
import java.awt.geom.Rectangle2D;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.*;
import java.util.List;
import java.util.concurrent.*;
import lombok.Getter;
import lombok.extern.log4j.Log4j2;
import org.apache.poi.sl.usermodel.TableCell;
import org.apache.poi.sl.usermodel.TextParagraph;
import org.apache.poi.sl.usermodel.TextShape;
import org.apache.poi.sl.usermodel.VerticalAlignment;
import org.apache.poi.util.Units;
import org.apache.poi.xddf.usermodel.XDDFLineProperties;
import org.apache.poi.xddf.usermodel.XDDFShapeProperties;
import org.apache.poi.xddf.usermodel.chart.*;
import org.apache.poi.xddf.usermodel.text.XDDFTextBody;
import org.apache.poi.xslf.usermodel.*;
import org.openxmlformats.schemas.drawingml.x2006.chart.*;
import org.springframework.http.HttpStatus;
import org.testcontainers.shaded.org.apache.commons.lang3.ArrayUtils;

import static app.valuationcontrol.multimodule.library.xlhandler.POICalcDocument.NUMBER_FORMAT;
import static app.valuationcontrol.multimodule.library.xlhandler.POICalcDocument.PERCENTAGE_FORMAT;

@Getter
@Log4j2
public class PresentationManager {

  XMLSlideShow slideShow;
  Model model;
  CalculationData calculationData;
  OpenAiServiceImplementation openAiServiceImplementation;
  final Double FONT_SIZE = 8D;
  final Double FULL_WIDTH = 640D;
  final Double FIRST_COLUMN = 120D;

  public PresentationManager(
      Model model,
      CalculationData calculationData,
      OpenAiServiceImplementation openAiServiceImplementation)
      throws IOException {

    InputStream templateFile =
        this.getClass().getClassLoader().getResourceAsStream("VC_Template.pptx");
    if (templateFile != null) {
      log.debug("Found template");
      this.slideShow = new XMLSlideShow(templateFile);
    } else {
      log.debug("Didn't find template");
      this.slideShow = new XMLSlideShow();
    }

    this.model = model;
    this.calculationData = calculationData;
    this.openAiServiceImplementation = openAiServiceImplementation;

    // Adding title
    XSLFSlideMaster slideMaster = this.slideShow.getSlideMasters().get(0);
    XSLFSlideLayout titleLayout = slideMaster.getLayout("TITLE_SLIDE");
    XSLFSlide slide = this.slideShow.createSlide(titleLayout);
    slide
        .getPlaceholder(0)
        .setText(
            "Valuation report for " + this.model.getCompany() + " - " + this.model.getStartYear());

    //Adding disclaimer
    titleLayout = slideMaster.getLayout("DISCLAIMER_SLIDE");
    slide = this.slideShow.createSlide(titleLayout);
    String templateText = "[Valuer] has been engaged by [SHORT_NAME]('[SHORT_NAME]', 'the Company') to conduct a valuation of the company. This report can only be used by [SHORT_NAME] and cannot be shared further without our written consent.\n" +
            "Valuation is a subjective exercise and may result in different outcomes under different assumptions. This report is based on our own choices, and when we specify a value, it is to be considered as a midpoint in a range. Estimates and forecasts are our own and not those of the company, and they should not be used in any context other than in this report.\n" +
            "The financial statements of [SHORT_NAME] for 2023 have not been audited, and we have assumed that, in all material respects, they represent a good starting point for a valuation. We have reflected this in our discount rate. [Valuer] is not responsible for the financial statements of [SHORT_NAME] and has not conducted a due diligence on them.\n";
    slide.getPlaceholder(0).setText("Disclaimer");
    slide.getPlaceholder(1).setText(templateText);


  }

  public void writeToByteArray(ByteArrayOutputStream byteArrayOutputStream) throws IOException {
    slideShow.write(byteArrayOutputStream);
  }

  private int addVariableToChart(
      HashMap<String, XDDFChartData> dataHashMap,
      XDDFChart chart,
      VariableData variableData,
      String graphType,
      XDDFCategoryAxis bottomAxis,
      XDDFValueAxis valueAxis,
      XDDFCategoryDataSource categoryData,
      int colOffset,
      boolean useSegment,boolean includeHistorical) {

    if (variableData != null) {

      int nbOfSeries = 1;
      if (variableData.modelledAtSegment()
          && !useSegment) { // If use segment we only create one serie with all segment values
        nbOfSeries = 1 + model.getSegments().size(); // Include total
      }
      boolean ignoreTotal = false;

      XDDFChartData data;
      if (dataHashMap.containsKey(graphType.toLowerCase())) {
        data = dataHashMap.get(graphType.toLowerCase());
      } else {
        if (graphType.equals("line")) {
          data = chart.createData(ChartTypes.LINE, bottomAxis, valueAxis);
        } else {
          data = chart.createData(ChartTypes.BAR, bottomAxis, valueAxis);
          ((XDDFBarChartData) data).setBarDirection(BarDirection.COL);
          ((XDDFBarChartData) data).setGapWidth(50);
          if (nbOfSeries == 1) {
            ((XDDFBarChartData) data).setBarGrouping(BarGrouping.CLUSTERED);
          } else {
            ((XDDFBarChartData) data).setBarGrouping(BarGrouping.STACKED);
            ((XDDFBarChartData) data).setOverlap((byte) 100);
          }
        }
        data.setVaryColors(false);
        dataHashMap.put(graphType.toLowerCase(), data);
      }
      // add chart values (y-axis data)

      int startAt = 0; // Include total segment
      if (nbOfSeries > 1 && graphType.equalsIgnoreCase("bar")) {
        startAt = 1; // Ignore total segment
      }

      for (int i = startAt; i < nbOfSeries; i++) {
        Double[] historicalValues;
        Double[] projectionValues;
        Double[] allValues;

        if(!useSegment){
        if (variableData.projectionValues() != null) {
          projectionValues = new Double[model.getNbProjectionPeriod()];
          for (int j = 0; j < variableData.projectionValues().get(i).length; j++) {
            try {
              projectionValues[j] = Double.parseDouble(String.valueOf(variableData.projectionValues().get(i)[j]));
            } catch (Exception e) {
              projectionValues[j] = null;
            }
          }
        }else{
          projectionValues=new Double[]{};
        }

        if(includeHistorical && variableData.historicalValues() != null){
          historicalValues = new Double[model.getNbHistoricalPeriod()];
          for (int j = 0; j < variableData.historicalValues().get(i).length; j++) {
            try {
              historicalValues[j] = Double.parseDouble(String.valueOf(variableData.historicalValues().get(i)[j]));
            } catch (Exception e) {
              historicalValues[j] = null;
            }
          }
        }else{
          historicalValues= new Double[]{};
        }

          allValues = Arrays.copyOf(historicalValues, projectionValues.length + historicalValues.length);
          System.arraycopy(projectionValues, 0, allValues, historicalValues.length, projectionValues.length);

        } else if(useSegment) {
          try {
            allValues =
                Arrays.copyOf(
                    variableData.singleOrConstantValue().toArray(),
                    model.getSegments().size() + 1,
                    Double[].class);
          } catch (ArrayStoreException e) {
            allValues = new Double[model.getSegments().size() + 1];
            Arrays.fill(allValues, 0D);
          }
        } else {
          throw new ResourceException(
              HttpStatus.INTERNAL_SERVER_ERROR,
              "Graph data is not valid for " + variableData.variableName());
        }

        String valuesDataRange =
            chart.formatRange(
                new org.apache.poi.ss.util.CellRangeAddress(
                    1, allValues.length, colOffset, colOffset));
        XDDFNumericalDataSource<Double> valueData =
            XDDFDataSourcesFactory.fromArray(allValues, valuesDataRange, colOffset);

        // adding series and defining legend
        XDDFChartData.Series series = data.addSeries(categoryData, valueData);

        String segmentName = i == 0 ? "Total" : model.getSegments().get(i - 1).getSegmentName();
        String title =
            nbOfSeries > 1
                ? variableData.variableName() + " " + segmentName
                : variableData.variableName();
        series.setTitle(title, chart.setSheetTitle(title, colOffset));
        colOffset = colOffset + 1;
        if (graphType.equals("bar")) {
          ((XDDFBarChartData.Series) series).setInvertIfNegative(false);
        }else{
          try{
            XDDFShapeProperties shapeProperties = Objects.requireNonNullElse(series.getShapeProperties(),new XDDFShapeProperties());
            XDDFLineProperties lineProperties = new XDDFLineProperties();
            lineProperties.setWidth(1D);
            shapeProperties.setLineProperties(lineProperties);
            series.setShapeProperties(shapeProperties);
            ((XDDFLineChartData.Series) series).setMarkerSize((short) 4);
          }catch (Exception e){
            log.debug(e);
          }
        }
      }
    }
    return colOffset;
  }

  public void createCharts(boolean includeAIcomments) {

    XSLFSlideMaster slideMaster = slideShow.getSlideMasters().get(0);
    XSLFSlideLayout titleLayout = slideMaster.getLayout("FIFTY_FIFTY_DIAGRAM");
    ArrayList<Future<JsonNode>> gptComments = new ArrayList<>();
    boolean allAiRequestsComplete = false;

    if (includeAIcomments) {
      try {
        // Limit to 1 concurrent requests at the same time
        ExecutorService executorService = Executors.newFixedThreadPool(2);
        // First thing is to ask for CHATGPT feedback
        for (ModelGraph graph : model.getGraphs()) {
          if (graph.getGraphVariable1Id() > 0) {
            if (model.getVariableWithID(graph.getGraphVariable1Id()).isPresent()) {
              Variable variable = model.getVariableWithID(graph.getGraphVariable1Id()).get();
              Objects.requireNonNull(variable);
              ObjectNode returnObject =
                  OpenAIHelperFunctions.prepareSingleVariable(
                      calculationData, model, variable, true);

              String prompt =
                  "You are a financial analyst describing a forecast. Do not list values but comment on their overall development. Please describe the financial development of ";
              if (variable.isModelledAtSegment() && variable.isSingleOrConstantValue()) {
                prompt = "Comment on the segment composition of ";
              } else if (variable.isSingleOrConstantValue()) {
                prompt =
                    "Using segmentValues and values (do not mention word 'segmentValues'), comment on the development of values and segment composition of ";
              }
              prompt = prompt + variable.getVariableName();

              ChatMessage systemMessage = new ChatMessage(ChatMessageRole.SYSTEM.value(), prompt);
              ChatMessage firstMsg =
                  new ChatMessage(ChatMessageRole.USER.value(), returnObject.toPrettyString());

              List<ChatMessage> messages = new ArrayList<>();
              messages.add(systemMessage);
              messages.add(firstMsg);
              Future<JsonNode> request =
                  executorService.submit(
                      () ->
                          OpenAIHelperFunctions.doRequest(
                              openAiServiceImplementation, messages, null, null));
              gptComments.add(request);
            }
          }
        }
        executorService.shutdown();
        allAiRequestsComplete = executorService.awaitTermination(1, TimeUnit.MINUTES);

      } catch (Exception e) {
        throw new ResourceException(
            HttpStatus.BAD_REQUEST, "Couldn't fetch comments for presentation");
      }
    }

    int index = 0;
    for (ModelGraph graph : model.getGraphs()) {
      HashMap<String, XDDFChartData> dataHashMap = new HashMap<>();
      if (graph.getGraphVariable1Id() > 0
          || graph.getGraphVariable2Id() > 0
          || graph.getGraphVariable3Id() > 0) {

        XSLFSlide slide = slideShow.createSlide(titleLayout);
        slide
            .getPlaceholder(0)
            .setText("Graph (" + (index + 1) + "/" + model.getGraphs().size() + ")");
        slide.getPlaceholder(1).setText(graph.getGraphName());
        slide.getPlaceholder(2).setText("Description");
        // create chart
        XSLFChart chart = slideShow.createChart();
        XDDFChartLegend legend = chart.getOrAddLegend();
        legend.setPosition(LegendPosition.BOTTOM);

        //Do not show blanks (Double with null)
        CTDispBlanksAs disp = CTDispBlanksAs.Factory.newInstance();
        disp.setVal(STDispBlanksAs.GAP);
        chart.getCTChart().setDispBlanksAs(disp);

        XDDFTextBody legendTextBody = new XDDFTextBody(legend);
        legendTextBody.getXmlObject().addNewBodyPr();
        legendTextBody.addNewParagraph().addDefaultRunProperties().setFontSize(FONT_SIZE);
        legend.setTextBody(legendTextBody);
        // legend.getTextBody().addNewParagraph().getDefaultRunProperties().setFontSize(FONT_SIZE);

        // set axis
        XDDFCategoryAxis leftBottomAxis = chart.createCategoryAxis(AxisPosition.BOTTOM);
        XDDFCategoryAxis rightBottomAxis = null;
        XDDFValueAxis leftAxis = chart.createValueAxis(AxisPosition.LEFT);
        XDDFValueAxis rightAxis = null;

        leftBottomAxis.crossAxis(leftAxis);
        leftBottomAxis.setTickLabelPosition(AxisTickLabelPosition.LOW);

        leftAxis.crossAxis(leftBottomAxis);
        leftAxis.setCrosses(AxisCrosses.AUTO_ZERO);
        leftAxis.setCrossBetween(AxisCrossBetween.BETWEEN);

        VariableData variableData1 =
            calculationData.getVariables().stream()
                .filter(v -> Objects.equals(v.id(), graph.getGraphVariable1Id()))
                .findFirst()
                .orElse(null);

        VariableData variableData2 =
            calculationData.getVariables().stream()
                .filter(v -> Objects.equals(v.id(), graph.getGraphVariable2Id()))
                .findFirst()
                .orElse(null);

        VariableData variableData3 =
            calculationData.getVariables().stream()
                .filter(v -> Objects.equals(v.id(), graph.getGraphVariable3Id()))
                .findFirst()
                .orElse(null);






        // Adding right and new bottom axis if needed
        if (graph.useTwoAxis()
                && (variableData1 != null
                    && graph.getGraphVariable1Axis().equalsIgnoreCase("secondary"))
            || (variableData2 != null
                && graph.getGraphVariable2Axis().equalsIgnoreCase("secondary"))
            || (variableData3 != null
                && graph.getGraphVariable3Axis().equalsIgnoreCase("secondary"))) {

          rightAxis = chart.createValueAxis(AxisPosition.RIGHT);
          rightAxis.setCrosses(AxisCrosses.MAX);

          rightAxis.setCrossBetween(AxisCrossBetween.BETWEEN);
          rightBottomAxis = chart.createCategoryAxis(AxisPosition.BOTTOM);
          rightBottomAxis.setVisible(false);

          rightBottomAxis.crossAxis(rightAxis);
          rightAxis.crossAxis(rightBottomAxis);
        }

        boolean useSegment = graph.useSegmentAxis();

        String[] categories;
        if (useSegment) {
          categories =
              ArrayUtils.addAll(
                  new String[] {"Total"},
                  model.getSegments().stream().map(Segment::getSegmentName).toArray(String[]::new));
        } else {
          int startOffset = graph.getGraphIncludeHistoricals() ? -model.getNbHistoricalPeriod() : 0;
          int arraySize = graph.getGraphIncludeHistoricals() ? model.getNbHistoricalPeriod()+model.getNbProjectionPeriod() : model.getNbProjectionPeriod();
            categories = new String[arraySize];
            // Creating value for X-Axis
            for (int i = startOffset; i < model.getNbProjectionPeriod(); i++) {
              categories[i-startOffset] = Integer.toString(model.getStartYear() + i);
            }
        }

        String categoryDataRange =
            chart.formatRange(
                new org.apache.poi.ss.util.CellRangeAddress(1, categories.length, 0, 0));
        XDDFCategoryDataSource categoryData =
            XDDFDataSourcesFactory.fromArray(categories, categoryDataRange, 0);

        XDDFValueAxis plotAxis;
        XDDFCategoryAxis bottomAxis;

        if (graph.useTwoAxis()
            && (graph.getGraphVariable1Axis().equalsIgnoreCase("secondary") && rightAxis != null)) {
          plotAxis = rightAxis;
          bottomAxis = rightBottomAxis;
        } else {
          plotAxis = leftAxis;
          bottomAxis = leftBottomAxis;
        }
        int colOffset = 1;
        colOffset =
            addVariableToChart(
                dataHashMap,
                chart,
                variableData1,
                graph.getGraphVariable1Type(),
                bottomAxis,
                plotAxis,
                categoryData,
                colOffset,
                useSegment,graph.getGraphIncludeHistoricals());

        // Plot variable2 and 3 only if variable 1 is not plotted using segment
        if (!useSegment) {
          if (graph.useTwoAxis()
              && (graph.getGraphVariable2Axis().equalsIgnoreCase("secondary")
                  && rightAxis != null)) {
            plotAxis = rightAxis;
            bottomAxis = rightBottomAxis;
          } else {
            plotAxis = leftAxis;
            bottomAxis = leftBottomAxis;
          }
          colOffset =
              addVariableToChart(
                  dataHashMap,
                  chart,
                  variableData2,
                  graph.getGraphVariable2Type(),
                  bottomAxis,
                  plotAxis,
                  categoryData,
                  colOffset,
                  false,graph.getGraphIncludeHistoricals());

          if (graph.useTwoAxis()
              && (graph.getGraphVariable3Axis().equalsIgnoreCase("secondary")
                  && rightAxis != null)) {
            plotAxis = rightAxis;
            bottomAxis = rightBottomAxis;
          } else {
            plotAxis = leftAxis;
            bottomAxis = leftBottomAxis;
          }
          addVariableToChart(
              dataHashMap,
              chart,
              variableData3,
              graph.getGraphVariable3Type(),
              bottomAxis,
              plotAxis,
              categoryData,
              colOffset,
              false,graph.getGraphIncludeHistoricals());
        }


        // Plotting data
        dataHashMap.forEach(
            (key, data) -> {
              data.getCategoryAxis().getOrAddTextProperties().setFontSize(FONT_SIZE);
              data.getValueAxes().forEach(v -> v.getOrAddTextProperties().setFontSize(FONT_SIZE));
              chart.plot(data);
            });

        try{
          leftAxis.setNumberFormat(NUMBER_FORMAT);
          CTValAx valAx = chart.getCTChart().getPlotArea().getValAxArray(0);
          if(variableData1!=null && variableData1.variableFormat().contains("percent")){
            valAx.getNumFmt().setFormatCode(PERCENTAGE_FORMAT);
            valAx.getNumFmt().setSourceLinked(false);
          }else{
            valAx.getNumFmt().setFormatCode(NUMBER_FORMAT);
            valAx.getNumFmt().setSourceLinked(false);
          }
        }catch (Exception e){
          log.info(e);
        }
        //Formatting Y axis, after data is populated

        // set chart dimensions !!Units are EMU (English Metric Units)!!
        Rectangle chartDimensions =
            new Rectangle(
                38 * Units.EMU_PER_POINT,
                95 * Units.EMU_PER_POINT,
                320 * Units.EMU_PER_POINT,
                180 * Units.EMU_PER_POINT);
        // add chart to slide
        slide.addChart(chart, chartDimensions);

        if (includeAIcomments && allAiRequestsComplete) {
          XSLFTextShape textShape = slide.getPlaceholder(3);
          try {
            JsonNode response = gptComments.get(index).get().get("response");
            textShape.setText(response.asText());
          } catch (InterruptedException | ExecutionException e) {
            throw new RuntimeException(e);
          }
        }
      }
      index = index + 1;
    }
  }

  private VariableData[] getVariableDataArrayFromScenarioComparison(
      ScenarioComparison comparison, Long variableId) {
    return comparison.comparisons().stream()
        .filter(c -> !c.isEmpty() && c.get(0).id().equals(variableId))
        .findFirst()
        .orElse(new ArrayList<>())
        .toArray(new VariableData[0]);
  }

  public void createComparison(ScenarioComparison comparison, boolean includeAIcomments) {
    XSLFSlideMaster slideMaster = slideShow.getSlideMasters().get(0);
    XSLFSlideLayout titleLayout = slideMaster.getLayout("FIFTY_FIFTY_DIAGRAM");

    VariableData[] variable1DataArray =
        getVariableDataArrayFromScenarioComparison(
            comparison, model.getKeyParam().getKeyOutput1Id());
    VariableData[] variable2DataArray =
        getVariableDataArrayFromScenarioComparison(
            comparison, model.getKeyParam().getKeyOutput2Id());

    Hashtable<Integer, VariableData[]> comparisons = new Hashtable<>();
    if (variable1DataArray.length > 0) comparisons.put(1, variable1DataArray);
    if (variable2DataArray.length > 0) comparisons.put(2, variable2DataArray);


    comparisons.forEach(

        (index, variableDataArray) -> {

          XSLFSlide slide = slideShow.createSlide(titleLayout);

          VariableData variableData = variableDataArray[0];

          slide.getPlaceholder(1).setText(variableData.variableName());
          slide.getPlaceholder(2).setText("Comparison of scenarios");

          XSLFChart chart = slideShow.createChart();
          // set axis
          XDDFCategoryAxis leftBottomAxis = chart.createCategoryAxis(AxisPosition.BOTTOM);
          XDDFValueAxis leftAxis = chart.createValueAxis(AxisPosition.LEFT);

          leftBottomAxis.crossAxis(leftAxis);
          leftAxis.crossAxis(leftBottomAxis);
          leftAxis.setCrosses(AxisCrosses.AUTO_ZERO);
          leftAxis.setCrossBetween(AxisCrossBetween.BETWEEN);

          // Creating value for X-Axis
          String[] categories =
              new String[] {
                SCENARIO.BASE.name(),
                SCENARIO.POSITIVE.name(),
                SCENARIO.NEGATIVE.name(),
                SCENARIO.SYNERGY.name(),
                SCENARIO.TEST.name()
              };
          String categoryDataRange =
              chart.formatRange(
                  new org.apache.poi.ss.util.CellRangeAddress(1, categories.length, 0, 0));
          XDDFCategoryDataSource categoryData =
              XDDFDataSourcesFactory.fromArray(categories, categoryDataRange, 0);

          ////////
          XDDFChartData data = chart.createData(ChartTypes.BAR, leftBottomAxis, leftAxis);
          ((XDDFBarChartData) data).setBarDirection(BarDirection.COL);
          ((XDDFBarChartData) data).setBarGrouping(BarGrouping.CLUSTERED);

          Double[] values;

          int periodToCheck =
              index.equals(1)
                  ? model.getKeyParam().getKeyOutput1Period()
                  : model.getKeyParam().getKeyOutput2Period();

          values =
              ArrayUtils.toObject(
                  Arrays.stream(variableDataArray)
                      .mapToDouble(
                          vd ->
                              Objects.requireNonNullElse(vd.getValueInPeriod(periodToCheck, 0), 0D))
                      .toArray());

          String valuesDataRange =
              chart.formatRange(
                  new org.apache.poi.ss.util.CellRangeAddress(1, values.length, 1, 1));
          XDDFNumericalDataSource<Double> valueData =
              XDDFDataSourcesFactory.fromArray(values, valuesDataRange, 1);

          // adding series and defining legend
          XDDFChartData.Series series = data.addSeries(categoryData, valueData);
          series.setTitle("Comparison 1", chart.setSheetTitle("Comparison 1", 1));
          ((XDDFBarChartData.Series) series).setInvertIfNegative(false);
          series.setShowLeaderLines(true);

          CTDLbls ctdLbls = chart
                  .getCTChart()
                  .getPlotArea()
                  .getBarChartArray(0)
                  .getSerArray(0)
                  .getDLbls();

          ctdLbls.addNewDLblPos()
              .setVal(STDLblPos.OUT_END);
          ctdLbls
              .addNewShowVal()
              .setVal(true);
          ctdLbls
              .addNewShowLegendKey()
              .setVal(false);
          ctdLbls
              .addNewShowCatName()
              .setVal(false);
          ctdLbls
              .addNewShowSerName()
              .setVal(false);

          chart.plot(data);

          Rectangle chartDimensions =
              new Rectangle(
                  38 * Units.EMU_PER_POINT,
                  95 * Units.EMU_PER_POINT,
                  320 * Units.EMU_PER_POINT,
                  180 * Units.EMU_PER_POINT);
          // add chart to slide
          slide.addChart(chart, chartDimensions);

          ObjectMapper objectMapper = new ObjectMapper();
          ObjectNode allDifferences = objectMapper.createObjectNode();
          ArrayNode positiveDifference = objectMapper.createArrayNode();
          ArrayNode negativeDifference = objectMapper.createArrayNode();
          ArrayNode synergiesDifference = objectMapper.createArrayNode();
          ArrayNode testDifference = objectMapper.createArrayNode();

          allDifferences.set(SCENARIO.POSITIVE.name(), positiveDifference);
          allDifferences.set(SCENARIO.NEGATIVE.name(), negativeDifference);
          allDifferences.set(SCENARIO.SYNERGY.name(), synergiesDifference);
          allDifferences.set(SCENARIO.TEST.name(), testDifference);

          //Getting relevant variableValues
          List <VariableValue> inputs =model.getVariableValuesUsedInScenarios();

          inputs.forEach(variableValue -> {
            try{
                Variable variable = variableValue.getAttachedVariable();
                List <VariableData>variableArray = comparison.comparisons().stream().filter(c -> c.get(0).id()==variable.getId()).findFirst().get();

                if (!variableArray.isEmpty()) {
                  boolean checkSingleEntryOnly = variable.isSingleOrConstantValue();
                  for (int i = 1;
                      i < variableArray.size();
                      i++) { // Checking differences to scenario 0

                    Integer segmentIndex = variable.isModelledAtSegment() ?  1+model.getSegments().indexOf(variableValue.getAttachedSegment()) : 0;
                    String segmentName = variableValue.getAttachedSegment()!=null ? variableValue.getAttachedSegment().getSegmentName(): "overall";

                    if (checkSingleEntryOnly) {
                      Objects.requireNonNull(variableArray.get(0).singleOrConstantValue());
                      Double baseScenarioValue =
                          OpenAIHelperFunctions.getDoubleFromObject(
                                  variableArray.get(0).singleOrConstantValue().get(segmentIndex), variableArray.get(0).variableFormat());
                      Double currentScenarioValue =
                          OpenAIHelperFunctions.getDoubleFromObject(
                                  variableArray.get(i).singleOrConstantValue().get(segmentIndex), variableArray.get(0).variableFormat());
                      // log.debug("{} vs {} for
                      // {}",baseScenarioValue,currentScenarioValue,variableArray[0].variableName());
                      if (baseScenarioValue != null
                          && !baseScenarioValue.equals(currentScenarioValue)) {
                        ObjectNode differenceNode = objectMapper.createObjectNode();
                        differenceNode.put("Variable", variable.getVariableName());
                        differenceNode.put("Segment",segmentName);
                        differenceNode.put("BaseValue", baseScenarioValue);
                        differenceNode.put("ScenarioValue", currentScenarioValue);
                        ArrayNode scenarioArray =
                            (ArrayNode) allDifferences.get(SCENARIO.from(i).name());
                        scenarioArray.add(differenceNode);
                        // differencesNode.add(differenceNode);
                      }
                    } else {
                      for (int j = 0; j < variableArray.get(i).projectionValues().get(segmentIndex).length; j++) {
                        Double baseScenarioValue =
                            OpenAIHelperFunctions.getDoubleFromObject(
                                    variableArray.get(0).projectionValues().get(segmentIndex)[j], variableArray.get(0).variableFormat());
                        Double currentScenarioValue =
                            OpenAIHelperFunctions.getDoubleFromObject(
                                    variableArray.get(i).projectionValues().get(segmentIndex)[j], variableArray.get(0).variableFormat());
                        // log.debug("{} vs {} for
                        // {}",baseScenarioValue,currentScenarioValue,variableArray[0].variableName());
                        if (baseScenarioValue != null
                            && !baseScenarioValue.equals(currentScenarioValue)) {
                          ObjectNode differenceNode = objectMapper.createObjectNode();
                          differenceNode.put("Variable", variable.getVariableName());
                          differenceNode.put("Year", model.getStartYear() + j);
                          differenceNode.put("Segment",segmentName);
                          differenceNode.put("BaseValue", baseScenarioValue);
                          differenceNode.put("ScenarioValue", currentScenarioValue);

                          ArrayNode scenarioArray =
                              (ArrayNode) allDifferences.get(SCENARIO.from(i).name());
                          scenarioArray.add(differenceNode);
                          // differencesNode.add(differenceNode);
                        }
                      }
                    }
                  }

                }}catch (Exception e){
                  log.debug(e);
                }
              });

          if (includeAIcomments) {
            String prompt =
                "Please comment on the differences between the scenarios, do not list variables but comment specifically for each scenario having variations to BaseValue";
            ChatMessage systemMessage = new ChatMessage(ChatMessageRole.SYSTEM.value(), prompt);
            ChatMessage firstMsg =
                new ChatMessage(ChatMessageRole.USER.value(), allDifferences.toPrettyString());

            List<ChatMessage> messages = new ArrayList<>();
            messages.add(systemMessage);
            messages.add(firstMsg);
            JsonNode request =
                OpenAIHelperFunctions.doRequest(openAiServiceImplementation, messages, null, null);
            slide.getPlaceholder(3).setText(request.get("response").asText());
          }
        });
  }

  public void addTextToCell(
      XSLFTableRow row,
      String text,
      Double fontSize,
      TextParagraph.TextAlign textAlign,
      boolean isTotal,
      boolean isHeader) {
    XSLFTableCell cell = row.addCell();
    XSLFTextRun textRun = cell.setText(text);
    textRun.setFontSize(fontSize);
    XSLFTextParagraph textParagraph = textRun.getParagraph();
    textParagraph.setTextAlign(textAlign);
    row.setHeight(15D);
    cell.setBottomInset(1D);
    cell.setTopInset(1D);
    cell.setLeftInset(0);
    cell.setRightInset(1D);
    cell.setVerticalAlignment(VerticalAlignment.MIDDLE);

    if (isTotal) {
      cell.setBorderWidth(TableCell.BorderEdge.bottom, 0.5D);
      cell.setBorderColor(TableCell.BorderEdge.bottom, Color.black);
      textRun.setBold(true);
    }

    if (isHeader) {
      cell.setFillColor(Color.LIGHT_GRAY);
      textRun.setBold(true);
    }
  }

  public void createTable(Long subAreaId, SCENARIO scenario, String slideTitle) {
    long numberOfVariablesInSubArea =
        this.calculationData.getVariables().stream()
            .filter(v -> v.variableSubAreaId().equals(subAreaId))
            .count();
    if (numberOfVariablesInSubArea > 0) {

      XSLFSlideMaster slideMaster = slideShow.getSlideMasters().get(0);
      XSLFSlideLayout titleLayout = slideMaster.getLayout("MASTER_SLIDE");
      XSLFSlide slide = slideShow.createSlide(titleLayout);
      slide.getPlaceholder(0).setText(slideTitle);

      XSLFTable table = slide.createTable();
      table.setAnchor(new Rectangle(38, 98, 100, 600));

      XSLFTableRow firstRow = table.addRow();
      addTextToCell(firstRow, "Item", FONT_SIZE, TextParagraph.TextAlign.LEFT, false, true);
      addTextToCell(firstRow, "Value", FONT_SIZE, TextParagraph.TextAlign.RIGHT, false, true);
      for (int i = 0; i < model.getNbHistoricalPeriod(); i++) {
        addTextToCell(
            firstRow,
            String.valueOf((model.getStartYear() - model.getNbHistoricalPeriod() + i)),
            FONT_SIZE,
            TextParagraph.TextAlign.RIGHT,
            false,
            true);
      }
      for (int i = 0; i < model.getNbProjectionPeriod(); i++) {
        addTextToCell(
            firstRow,
            String.valueOf((model.getStartYear() + i)),
            FONT_SIZE,
            TextParagraph.TextAlign.RIGHT,
            false,
            true);
      }

      this.calculationData.getVariables().stream()
          .filter(v -> v.variableSubAreaId().equals(subAreaId))
          .forEach(
              v -> {
                XSLFTableRow row = table.addRow();
                boolean isTotal = v.variableType().toLowerCase().contains("total");

                addTextToCell(
                    row, v.variableName(), FONT_SIZE, TextParagraph.TextAlign.LEFT, isTotal, false);

                if (v.isSingleEntry()) {
                  String value = null;
                  if (v.singleOrConstantValue() != null) {
                    value =
                        (!v.singleOrConstantValue().isEmpty())
                            ? OpenAIHelperFunctions.getFormattedValueAsString(
                                v.singleOrConstantValue().get(scenario.ordinal()), v)
                            : null;
                  }
                  addTextToCell(
                      row,
                      Objects.requireNonNullElse(value, " "),
                      FONT_SIZE,
                      TextParagraph.TextAlign.RIGHT,
                      isTotal,
                      false);
                } else {
                  addTextToCell(row, " ", FONT_SIZE, TextParagraph.TextAlign.RIGHT, isTotal, false);
                }
                for (int i = 0; i < model.getNbHistoricalPeriod(); i++) {
                  String value = null;
                  if (v.historicalValues() != null) {
                    value =
                        (!v.historicalValues().isEmpty())
                            ? OpenAIHelperFunctions.getFormattedValueAsString(
                                v.historicalValues().get(scenario.ordinal())[i], v)
                            : null;
                  }
                  addTextToCell(
                      row,
                      Objects.requireNonNullElse(value, " "),
                      FONT_SIZE,
                      TextParagraph.TextAlign.RIGHT,
                      isTotal,
                      false);
                }
                for (int i = 0; i < model.getNbProjectionPeriod(); i++) {
                  String value = null;
                  if (v.projectionValues() != null) {
                    value =
                        (!v.projectionValues().isEmpty())
                            ? OpenAIHelperFunctions.getFormattedValueAsString(
                                v.projectionValues().get(scenario.ordinal())[i], v)
                            : null;
                  }
                  addTextToCell(
                      row,
                      Objects.requireNonNullElse(value, " "),
                      FONT_SIZE,
                      TextParagraph.TextAlign.RIGHT,
                      isTotal,
                      false);
                }
              });
      table.setColumnWidth(0, FIRST_COLUMN);
      int nbOfColumns = 1 + model.getNbHistoricalPeriod() + model.getNbProjectionPeriod();
      double columnWidth = (FULL_WIDTH - FIRST_COLUMN) / nbOfColumns;
      for (int i = 1; i <= nbOfColumns; i++) {
        table.setColumnWidth(i, columnWidth);
      }
    }
  }

  public void createSensitivities(boolean includeAIcomments) {

    int chunkSize = 2;
    boolean hasSegment =
        calculationData.getSensitivities().stream()
            .reduce(
                false,
                (result, s) ->
                    model
                        .getVariableWithID(s.measurementVariableId())
                        .orElse(new Variable())
                        .isModelledAtSegment(),
                Boolean::logicalAnd);

    double numberOfSegments = hasSegment ? model.getSegments().size() : 0;
    double numberOfChunks =
        Math.ceil((double) calculationData.getSensitivities().size() / chunkSize);

    XSLFSlideMaster slideMaster = slideShow.getSlideMasters().get(0);
    XSLFSlideLayout titleLayout = slideMaster.getLayout("FIFTY_FIFTY_SUBTITLE_ONLY");

    for (int segIndex = 0; segIndex <= numberOfSegments; segIndex++) {
      XSLFSlide slide = slideShow.createSlide(titleLayout);
      String segmentTitle =
          segIndex > 0
              ? " for segment " + calculationData.getSegments().get(segIndex - 1).segmentName()
              : "";
      slide
          .getPlaceholder(0)
          .setText(
              "Sensitivities (1/" + Double.valueOf(numberOfChunks).intValue() + ")" + segmentTitle);

      for (int i = 0; i < calculationData.getSensitivities().size(); i++) {
        if (i > 0 && i % 2 == 0) {
          double chunkIndex = Math.ceil((double) (i + 1) / 2);
          slide = slideShow.createSlide(titleLayout);
          slide
              .getPlaceholder(0)
              .setText(
                  "Sensitivities ("
                      + Double.valueOf(chunkIndex).intValue()
                      + "/"
                      + Double.valueOf(numberOfChunks).intValue()
                      + ")"
                      + segmentTitle);
        }
        int oneTwoIndex = (i % 2) + 1;

        SensitivityData sensitivity = calculationData.getSensitivities().get(i);
        SensitivityResult sensitivityResult = calculationData.getSensitivityResults().get(i);
        VariableData measurement =
            calculationData.getVariables().stream()
                .filter(v -> v.id().equals(sensitivity.measurementVariableId()))
                .findFirst()
                .orElseThrow(
                    () ->
                        new ResourceException(
                            HttpStatus.INTERNAL_SERVER_ERROR, "Could find sensitivity variable"));
        VariableData variable1 =
            calculationData.getVariables().stream()
                .filter(v -> v.id().equals(sensitivity.variable1Id()))
                .findFirst()
                .orElseThrow(
                    () ->
                        new ResourceException(
                            HttpStatus.INTERNAL_SERVER_ERROR, "Could find sensitivity variable"));
        VariableData variable2 =
            calculationData.getVariables().stream()
                .filter(v -> v.id().equals(sensitivity.variable2Id()))
                .findFirst()
                .orElseThrow(
                    () ->
                        new ResourceException(
                            HttpStatus.INTERNAL_SERVER_ERROR, "Could find sensitivity variable"));

        if (hasSegment && !measurement.modelledAtSegment()) {
          throw new ResourceException(
              HttpStatus.INTERNAL_SERVER_ERROR, "Error in the use of segment in sensitivities");
        }

        // Creating a new table
        XSLFTable table = slide.createTable();

        // Creating header row for variable2
        XSLFTableRow row = table.addRow();

        addTextToCell(row, "", FONT_SIZE, TextParagraph.TextAlign.LEFT, false, true);
        addTextToCell(row, "", FONT_SIZE, TextParagraph.TextAlign.LEFT, false, true);
        addTextToCell(
            row, variable2.variableName(), FONT_SIZE, TextParagraph.TextAlign.CENTER, false, true);
        for (int j = 1; j < sensitivityResult.getVariable2Values().length; j++) {
          addTextToCell(row, "", FONT_SIZE, TextParagraph.TextAlign.CENTER, false, true);
        }
        row.mergeCells(2, 2 + sensitivityResult.getVariable2Values().length - 1);
        // Adding Values2 row
        row = table.addRow();
        addTextToCell(row, "", FONT_SIZE, TextParagraph.TextAlign.LEFT, false, true);
        addTextToCell(row, "", FONT_SIZE, TextParagraph.TextAlign.LEFT, false, true);
        for (int j = 0; j < sensitivityResult.getVariable2Values().length; j++) {
          addTextToCell(
              row,
              OpenAIHelperFunctions.getFormattedValueAsString(
                  sensitivityResult.getVariable2Values()[j], variable2),
              FONT_SIZE,
              TextParagraph.TextAlign.RIGHT,
              false,
              true);
        }

        for (int j = 0; j < sensitivityResult.getVariable1Values().length; j++) {
          row = table.addRow();
          if (j == 0) { // Adding header
            addTextToCell(
                row,
                variable1.variableName(),
                FONT_SIZE,
                TextParagraph.TextAlign.CENTER,
                false,
                true);
          } else {
            addTextToCell(row, "", FONT_SIZE, TextParagraph.TextAlign.CENTER, false, true);
          }
          //
          addTextToCell(
              row,
              OpenAIHelperFunctions.getFormattedValueAsString(
                  sensitivityResult.getVariable1Values()[j], variable1),
              FONT_SIZE,
              TextParagraph.TextAlign.RIGHT,
              false,
              true);
          for (int k = 0; k < sensitivityResult.getVariable2Values().length; k++) {
            String value =
                OpenAIHelperFunctions.getFormattedValueAsString(
                    sensitivityResult.getMyResultArray()[segIndex][j][k], measurement);
            addTextToCell(row, value, FONT_SIZE, TextParagraph.TextAlign.RIGHT, false, false);
          }
        }

        slide.getPlaceholder(oneTwoIndex).setText(sensitivity.description());
        Rectangle2D anchor = titleLayout.getPlaceholder(oneTwoIndex).getAnchor();
        table.setAnchor(new Rectangle((int) anchor.getX(), (int) anchor.getY() + 45, 0, 0));

        table.mergeCells(2, 2 + sensitivityResult.getVariable1Values().length - 1, 0, 0);
        table.getCell(2, 0).setTextDirection(TextShape.TextDirection.VERTICAL_270);
        table.getCell(2, 0).setVerticalAlignment(VerticalAlignment.MIDDLE);
        table.setColumnWidth(0, 30);
        table.setColumnWidth(1, 40);
        double column_width = 246D / sensitivityResult.getVariable2Values().length;
        for (int l = 0; l < sensitivityResult.getVariable2Values().length; l++) {
          table.setColumnWidth(2 + l, column_width);
        }
      } // End for each sensitivity
    } // End for each segment
  }

  public void createSummary() {
    // Adding title
    XSLFSlideMaster slideMaster = this.slideShow.getSlideMasters().get(0);
    XSLFSlideLayout titleLayout = slideMaster.getLayout("FIFTY_FIFTY_TEXT");
    XSLFSlide slide = this.slideShow.createSlide(titleLayout);

    String templateText =
        "This report is a valuation report for %s. "
            + "It is built on a DCF model using a %s years projection period. "
            + "While the valuation is presented as a single value, it should be read as the mid-point estimate in a range. Refer to sensitivities for more details.";
    String text = String.format(templateText, model.getCompany(), model.getNbProjectionPeriod());
    slide.getPlaceholder(0).setText("Disclaimer");
    slide.getPlaceholder(1).setText("Methodology");
    slide.getPlaceholder(2).setText("Key Assumptions");
    slide.getPlaceholder(3).setText(text);

    VariableData keyOutputVariable1 =
        calculationData.getVariables().stream()
            .filter(v -> v.id().equals(model.getKeyParam().getKeyOutput1Id()))
            .findFirst()
            .orElse(null);
    VariableData keyOutputVariable2 =
        calculationData.getVariables().stream()
            .filter(v -> v.id().equals(model.getKeyParam().getKeyOutput2Id()))
            .findFirst()
            .orElse(null);
    VariableData keyParam1Variable =
        calculationData.getVariables().stream()
            .filter(v -> v.id().equals(model.getKeyParam().getKeyParam1Id()))
            .findFirst()
            .orElse(null);
    VariableData keyParam2Variable =
        calculationData.getVariables().stream()
            .filter(v -> v.id().equals(model.getKeyParam().getKeyParam2Id()))
            .findFirst()
            .orElse(null);
    VariableData keyParam3Variable =
        calculationData.getVariables().stream()
            .filter(v -> v.id().equals(model.getKeyParam().getKeyParam3Id()))
            .findFirst()
            .orElse(null);
    VariableData keyParam4Variable =
        calculationData.getVariables().stream()
            .filter(v -> v.id().equals(model.getKeyParam().getKeyParam4Id()))
            .findFirst()
            .orElse(null);

    String keyOutputVariable1Value =
        keyOutputVariable1 != null
            ? OpenAIHelperFunctions.getFormattedValueAsString(
                keyOutputVariable1.getValueInPeriod(model.getKeyParam().getKeyOutput1Period(), 0),
                keyOutputVariable1)
            : "";
    String keyOutputVariable2Value =
        keyOutputVariable2 != null
            ? OpenAIHelperFunctions.getFormattedValueAsString(
                keyOutputVariable2.getValueInPeriod(model.getKeyParam().getKeyOutput2Period(), 0),
                keyOutputVariable2)
            : "";
    String keyParam1VariableValue =
        keyParam1Variable != null
            ? OpenAIHelperFunctions.getFormattedValueAsString(
                keyParam1Variable.getValueInPeriod(model.getKeyParam().getKeyParam1Period(), 0),
                keyParam1Variable)
            : "";
    String keyParam2VariableValue =
        keyParam2Variable != null
            ? OpenAIHelperFunctions.getFormattedValueAsString(
                keyParam2Variable.getValueInPeriod(model.getKeyParam().getKeyParam2Period(), 0),
                keyParam2Variable)
            : "";
    String keyParam3VariableValue =
        keyParam3Variable != null
            ? OpenAIHelperFunctions.getFormattedValueAsString(
                keyParam3Variable.getValueInPeriod(model.getKeyParam().getKeyParam3Period(), 0),
                keyParam3Variable)
            : "";
    String keyParam4VariableValue =
        keyParam4Variable != null
            ? OpenAIHelperFunctions.getFormattedValueAsString(
                keyParam4Variable.getValueInPeriod(model.getKeyParam().getKeyParam4Period(), 0),
                keyParam4Variable)
            : "";

    XSLFTextParagraph textParagraph = slide.getPlaceholder(4).getTextParagraphs().get(0);
    XSLFTextRun textRun = textParagraph.addNewTextRun();
    text = "The key outputs of the model are:";
    textRun.setText(text);

    textParagraph = slide.getPlaceholder(4).addNewTextParagraph();
    textParagraph.setBullet(true);
    textParagraph.setIndent(1D);
    text =
        (keyOutputVariable1 != null
                ? keyOutputVariable1.variableName() + " : " + keyOutputVariable1Value + "\n "
                : "")
            + (keyOutputVariable2 != null
                ? keyOutputVariable2.variableName() + " : " + keyOutputVariable2Value
                : "");
    textRun = textParagraph.addNewTextRun();
    textRun.setText(text);

    textParagraph = slide.getPlaceholder(4).addNewTextParagraph();
    textRun = textParagraph.addNewTextRun();
    textRun.setText("\n\nThe main value drivers for these values are: ");

    textParagraph = slide.getPlaceholder(4).addNewTextParagraph();
    textParagraph.setBullet(true);
    textParagraph.setIndent(1D);
    text =
        (keyParam1Variable != null
                ? keyParam1Variable.variableName() + " : " + keyParam1VariableValue + "\n "
                : "")
            + (keyParam2Variable != null
                ? keyParam2Variable.variableName() + " : " + keyParam2VariableValue + "\n "
                : "")
            + (keyParam3Variable != null
                ? keyParam3Variable.variableName() + " : " + keyParam3VariableValue + "\n "
                : "")
            + (keyParam4Variable != null
                ? keyParam4Variable.variableName() + " : " + keyParam4VariableValue
                : "");
    textRun = textParagraph.addNewTextRun();
    textRun.setText(text);
  }
}