package app.valuationcontrol.multimodule.library.entities;

import app.valuationcontrol.multimodule.library.helpers.DataTransformer;
import app.valuationcontrol.multimodule.library.helpers.ModelProvider;
import app.valuationcontrol.multimodule.library.records.ModelData;
import jakarta.persistence.CascadeType;
import jakarta.persistence.Embedded;
import jakarta.persistence.Entity;
import jakarta.persistence.FetchType;
import jakarta.persistence.GeneratedValue;
import jakarta.persistence.GenerationType;
import jakarta.persistence.Id;
import jakarta.persistence.OneToMany;
import jakarta.persistence.OrderBy;
import jakarta.persistence.Transient;
import java.util.*;
import java.util.stream.Collectors;
import lombok.Getter;
import lombok.Setter;
import org.hibernate.annotations.ColumnDefault;
import org.testcontainers.shaded.org.apache.commons.lang3.RandomStringUtils;

/**
 * This class is an entity that manages a model
 *
 * @author thomas
 */
@Getter
@Entity
public class Model implements DataTransformer<ModelData>, ModelProvider {

  @Id
  @GeneratedValue(strategy = GenerationType.IDENTITY)
  private Long id;

  @Setter Long templateId = -1L;
  @Setter private String name;
  @Setter private Integer startYear;
  @Setter private Integer nbHistoricalPeriod;
  @Setter private Integer nbProjectionPeriod;
  @Setter private String company;

  @Setter private String companyNumber;
  @Setter private String currency;
  @Setter private String simpleModel;

  @Setter
  @ColumnDefault("false")
  private boolean locked;

  @Setter
  @ColumnDefault("false")
  private boolean includeYTD;

  @Setter @Embedded private KeyParam keyParam;

  @OneToMany(
      mappedBy = "attachedModel",
      fetch = FetchType.LAZY,
      cascade = CascadeType.ALL,
      orphanRemoval = true)
  @OrderBy(value = "id ASC")
  private final List<Variable> variables = new ArrayList<>();

  @OneToMany(
      mappedBy = "attachedModel",
      fetch = FetchType.LAZY,
      cascade = CascadeType.ALL,
      orphanRemoval = true)
  private final List<LogEntry> logEntries = new ArrayList<>();

  @OneToMany(
      mappedBy = "attachedModel",
      fetch = FetchType.LAZY,
      cascade = CascadeType.ALL,
      orphanRemoval = true)
  @OrderBy(value = "id ASC")
  private final List<Segment> segments = new ArrayList<>();

  @OneToMany(
      mappedBy = "attachedModel",
      fetch = FetchType.LAZY,
      cascade = CascadeType.ALL,
      orphanRemoval = true)
  @OrderBy(value = "areaOrder ASC")
  private final List<Area> areas = new ArrayList<>();

  @OneToMany(
      mappedBy = "attachedModel",
      fetch = FetchType.LAZY,
      cascade = CascadeType.ALL,
      orphanRemoval = true)
  private final List<ModelGraph> graphs = new ArrayList<>();

  @OneToMany(
      mappedBy = "attachedModel",
      fetch = FetchType.LAZY,
      cascade = CascadeType.ALL,
      orphanRemoval = true)
  private final List<Sensitivity> sensitivities = new ArrayList<>();

  @Transient
  private Integer colBeforeDataStart =
      10; // Indicates the column in Excel where the data starts

  @Transient
  private Boolean useAreaAddress =
      false; // Set to true to get an address based on Area and Variable Order

  @Transient private final HashMap<Area, String> areaStringHashMap = new HashMap<>();

  /**
   * @param modelName is the areaName of the new model to be created
   * @param numberOfHistoricalPeriods indicates the number of historical periods to be managed in
   *     the model
   * @param numberOfProjectionPeriod indicates the number of projection periods in the model
   */
  public Model(
      String modelName, Integer numberOfHistoricalPeriods, Integer numberOfProjectionPeriod) {
    this.name = modelName;
    this.setNbHistoricalPeriod(numberOfHistoricalPeriods);
    this.setNbProjectionPeriod(numberOfProjectionPeriod);
  }

  public Model(ModelData modelData) {
    this.name = modelData.name();
    this.currency = modelData.currency();
    this.company = modelData.company();
    this.companyNumber = modelData.companyNumber();
    this.nbProjectionPeriod = modelData.nbProjectionPeriod();
    this.nbHistoricalPeriod = modelData.nbHistoricalPeriod();
    this.startYear = modelData.startYear();
    this.keyParam = modelData.keyParam();
    this.includeYTD = modelData.includeYTD();
  }

  public Model() {}

  public Integer getArraySize() {
    /*We use constant column as it corresponds to the first column with data, the columns before are metadata*/
    int metadataSize = this.getConstantColumn();
    // 1 for constant column  + (1 since we also have main segment + number of segments) * (number
    // of
    // historical + number of projections)
    int dataSize = this.getNumberOfPeriodsAndValue() * (this.getSegments().size() + 1);
    return metadataSize + dataSize;
  }

  public int indexOfLastColumn() {
    int segmentColumnOffset = this.getNumberOfPeriodsAndValue();
    return this.getTotalNumberOfColumns(true) + this.getSegments().size() * segmentColumnOffset;
  }

  public int getConstantColumn(){
    return colBeforeDataStart-1;
  }
  public void setUseAreaAddress(boolean useAreaAddress) {

    if (useAreaAddress) {
      this.colBeforeDataStart=4; //set to 1 to only includes values
      this.getAreaStringHashMap().clear();
      this.getAreas()
          .forEach(
              area -> {
                String randomString = RandomStringUtils.randomAlphanumeric(8);
                this.getAreaStringHashMap().put(area, randomString);
                area.getSubAreas()
                    .sort(
                        Comparator.comparingInt(
                            sa ->
                                Objects.requireNonNullElse(
                                    sa.getSubAreaOrder(), 1))); // Sorting subAreas
              });

      // Sorting variables
      this.getVariables()
          .sort(Comparator.comparingInt(v -> Objects.requireNonNullElse(v.getVariableOrder(), 1)));
      this.getVariables()
          .sort(
              Comparator.comparingInt(
                  v -> Objects.requireNonNullElse(v.getVariableSubArea().getSubAreaOrder(), 1)));
    } else {
      this.getVariables().sort(Comparator.comparingLong(Variable::getId));
    }
    this.useAreaAddress = useAreaAddress;
  }

  public Integer getFirstProjectionColumn() {
    return colBeforeDataStart + this.nbHistoricalPeriod;
  }

  public int getTotalNumberOfColumns(boolean startingAtZero) {
    if (startingAtZero) {
      return colBeforeDataStart + this.getNumberOfPeriodsAndValue() - 2; // start at 1
    } else {
      return colBeforeDataStart
          + this.getNumberOfPeriodsAndValue()
          - 1; // less the constant value column
    }
  }

  public int getNumberOfPeriodsAndValue() {
    int nbAdditionalValues = 1;
    if (this.includeYTD) {
      nbAdditionalValues = 3;
    }

    return nbHistoricalPeriod + nbProjectionPeriod + nbAdditionalValues;
  }

  public int getNumberOfVariables() {
    return this.getVariables().size();
  }

  /*Getters and setters */

  /**
   * Iterates through all existing variables and return the highest existing variable areaOrder
   *
   * @return the highest existing variable_order
   */
  public Integer getMaxVariableOrder() {
    return this.getVariables().stream()
        .map(Variable::getVariableOrder)
        .filter(Objects::nonNull)
        .max(Integer::compareTo)
        .orElse(0);
  }

  public Optional<Variable> getVariableWithID(long variableId) {
    return this.getVariables().stream()
        .filter(existingVariable -> existingVariable.getId() == variableId)
        .findFirst();
  }

  public ModelData asData() {
    /*Other dimensions are returned by CalculationData*/
    KeyParam keyParamCopy = new KeyParam();
    keyParamCopy.copyFromExisting(this.getKeyParam(), this);

    keyParamCopy.setKeyOutput1Period(
        Sensitivity.asModelYear(
            this.getKeyParam() != null ? this.getKeyParam().getKeyOutput1Period() : null, this));
    keyParamCopy.setKeyOutput2Period(
        Sensitivity.asModelYear(
            this.getKeyParam() != null ? this.getKeyParam().getKeyOutput2Period() : null, this));
    keyParamCopy.setKeyParam1Period(
        Sensitivity.asModelYear(
            this.getKeyParam() != null ? this.getKeyParam().getKeyParam1Period() : null, this));
    keyParamCopy.setKeyParam2Period(
        Sensitivity.asModelYear(
            this.getKeyParam() != null ? this.getKeyParam().getKeyParam2Period() : null, this));
    keyParamCopy.setKeyParam3Period(
        Sensitivity.asModelYear(
            this.getKeyParam() != null ? this.getKeyParam().getKeyParam3Period() : null, this));
    keyParamCopy.setKeyParam4Period(
        Sensitivity.asModelYear(
            this.getKeyParam() != null ? this.getKeyParam().getKeyParam4Period() : null, this));

    return new ModelData(
        id,
        name,
        startYear,
        nbHistoricalPeriod,
        nbProjectionPeriod,
        company,
        companyNumber,
        currency,
        simpleModel,
        locked,
        includeYTD,
        keyParamCopy);
  }

  @Override
  public Long getModelId() {
    return this.getId();
  }

  // For test only
  public void forceModelId(Long modelId) {
    this.id = modelId;
  }

  public Set<Long> getKeyVariablesForModel() {

    Set<Long> ids = new HashSet<>();

    if (this.getKeyParam() != null) {
      if (this.getKeyParam().getKeyOutput1Id() != null) {
        ids.add(this.getKeyParam().getKeyOutput1Id());
      }

      if (this.getKeyParam().getKeyOutput2Id() != null) {
        ids.add(this.getKeyParam().getKeyOutput2Id());
      }

      if (this.getKeyParam().getKeyParam1Id() != null) {
        ids.add(this.getKeyParam().getKeyParam1Id());
      }
      if (this.getKeyParam().getKeyParam2Id() != null) {
        ids.add(this.getKeyParam().getKeyParam2Id());
      }
      if (this.getKeyParam().getKeyParam3Id() != null) {
        ids.add(this.getKeyParam().getKeyParam3Id());
      }
      if (this.getKeyParam().getKeyParam4Id() != null) {
        ids.add(this.getKeyParam().getKeyParam4Id());
      }
    }

    this.getGraphs()
        .forEach(
            modelGraph -> {
              if (modelGraph.getGraphVariable1Id() != null) {
                ids.add(modelGraph.getGraphVariable1Id());
              }
              if (modelGraph.getGraphVariable2Id() != null) {
                ids.add(modelGraph.getGraphVariable2Id());
              }
              if (modelGraph.getGraphVariable3Id() != null) {
                ids.add(modelGraph.getGraphVariable3Id());
              }
            });

    this.getSensitivities()
        .forEach(
            sensitivity -> {
              if (sensitivity.getSensitivityMeasurementVariableId() != null) {
                ids.add(sensitivity.getSensitivityMeasurementVariableId());
              }
              if (sensitivity.getSensitivityVariable1Id() != null) {
                ids.add(sensitivity.getSensitivityVariable1Id());
              }
              if (sensitivity.getSensitivityVariable2Id() != null) {
                ids.add(sensitivity.getSensitivityVariable2Id());
              }
            });

    this.getVariables().stream()
        .filter(
            variable ->
                variable.getVariableValues() != null
                    && variable.getVariableValues().stream()
                        .anyMatch(variableValue -> variableValue.getScenarioNumber() > 0))
        .forEach(variable -> ids.add(variable.getId()));

    return ids.stream().filter(aLong -> aLong > -1L).collect(Collectors.toSet());
  }


  public List<VariableValue> getVariableValuesUsedInScenarios(){
      return this.getVariables().stream().flatMap(variable -> variable.getVariableValues().stream().filter(vv->vv.getScenarioNumber()>0)).toList();
  }



}
