package app.valuationcontrol.multimodule.library.entities;

import static java.util.List.of;

import app.valuationcontrol.multimodule.library.helpers.*;
import app.valuationcontrol.multimodule.library.records.SubAreaData;
import app.valuationcontrol.multimodule.library.records.VariableData;
import app.valuationcontrol.multimodule.library.xlhandler.CellValueHelper;
import app.valuationcontrol.multimodule.library.xlhandler.SCENARIO;
import app.valuationcontrol.multimodule.library.xlhandler.XLInstance;
import jakarta.persistence.*;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import lombok.Getter;
import lombok.Setter;

/**
 * This class is an entity that a variable within a model
 *
 * @author thomas
 */
@Getter
@Setter
@Entity
@Table(uniqueConstraints = {@UniqueConstraint(columnNames = {"attached_model_id", "variableName"})})
public class Variable implements DataTransformer<VariableData>, ModelProvider {
  public static final String CONSTANT = "constant";
  private static final String SINGLE_RESULT = "single_result";
  public static final List<String> CONSTANT_OR_SINGLE = List.of(CONSTANT, SINGLE_RESULT);
  private static final List<String> NO_VALUE_VARIABLE_TYPES = of("sub_total_row", "total_row");

  @Id
  @GeneratedValue(strategy = GenerationType.IDENTITY)
  private Long id;

  private String variableName;

  private String variableDescription;

  @Column(length = 500)
  private String variableFormula;

  private String variableType;

  private Integer variableOrder;
  private Integer variableDepth;
  private boolean variableApplyToHistoricals;
  private String variableFormat;

  @Column(length = 500)
  private String evaluatedVariableFormula;

  @ManyToOne private Area variableArea;

  @ManyToOne private SubArea variableSubArea;

  @ManyToOne private Model attachedModel;

  @OneToMany(
      mappedBy = "attachedVariable",
      fetch = FetchType.EAGER,
      cascade = CascadeType.ALL,
      orphanRemoval = true)
  private List<VariableValue> variableValues = new ArrayList<>();

  @ManyToMany(fetch = FetchType.EAGER, cascade = CascadeType.ALL)
  private List<Variable> variableDependencies = new ArrayList<>();

  @Transient private Long originalId;

  @Transient private List<Object[]> historicalValues;

  @Transient private List<Object[]> projectionValues;

  @Transient
  private List<Object> singleOrConstantValue =
      new ArrayList<>(); // Stores the current value of a singleValue or the value of a

  @Transient
  private final List<Object> yearToDateValue =
      new ArrayList<>(); // Stores the current value of a singleValue or the value of a

  @Transient
  private final List<Object> yearToGoValue =
      new ArrayList<>(); // Stores the current value of a singleValue or the value of a

  @Transient
  private final List<VariableReplacementOccurrence> replacementOccurrences = new ArrayList<>();

  public Variable() {}

  public Variable(Variable existingVariable) {
    // Copying all attributes apart from id
    this.variableName = existingVariable.getVariableName();
    this.variableDescription = existingVariable.getVariableDescription();
    this.variableFormula = existingVariable.getVariableFormula();
    this.variableType = existingVariable.getVariableType();
    this.variableFormat = existingVariable.getVariableFormat();
    this.variableApplyToHistoricals = existingVariable.isVariableApplyToHistoricals();
    this.variableDepth = existingVariable.getVariableDepth();
    this.variableOrder =
        existingVariable.getVariableOrder(); // Keep same areaOrder when replicating a variable

    // Need to override in case of a variable copy without same ID
    this.evaluatedVariableFormula = existingVariable.getEvaluatedVariableFormula();
    this.variableDependencies = existingVariable.getVariableDependencies();
    this.variableValues = existingVariable.getVariableValues();
    this.variableSubArea = existingVariable.getVariableSubArea();
    this.variableArea = this.variableSubArea.getAttachedArea();
    this.attachedModel = existingVariable.getAttachedModel();
  }

  public Variable(VariableData variableData, Model model, Area area, SubArea subArea) {
    this.attachedModel = model;
    this.updateFromVariableData(variableData, area, subArea);
    this.setVariableOrder(model.getMaxVariableOrder() + 1);
  }

  public Variable(Variable templateVariable, SubArea variableSubArea) {
    this(templateVariable);
    // Override values
    this.evaluatedVariableFormula = null; // Need to be null to be evaluated
    this.variableDependencies = new ArrayList<>();
    this.variableValues = new ArrayList<>();
    this.variableSubArea = variableSubArea;
    this.variableArea = variableSubArea.getAttachedArea();
    this.attachedModel = variableSubArea.getAttachedArea().getAttachedModel();
  }

  public void updateFromVariableData(VariableData variableData, Area area, SubArea subArea) {

    this.setVariableFormula(variableData.variableFormula());
    // Evaluate the new formula (Updating name afterward so that user does not need to also updated
    // formula when changing name
    this.setEvaluatedVariableFormula(FormulaEvaluator.evaluateVariableFormula(this));

    // Managing variableValues based on type
    manageVariableValues(
        this.getVariableValues(), this.getVariableType(), variableData.variableType());

    // Updating type
    this.setVariableType(variableData.variableType());
    // Updating name after formula update so
    this.setVariableName(variableData.variableName().trim());
    this.setVariableDescription(variableData.variableDescription());
    this.setVariableDepth(variableData.variableDepth());
    this.setVariableApplyToHistoricals(variableData.variableApplyToHistoricals());
    this.setVariableFormat(variableData.variableFormat());
    this.setVariableArea(area);
    this.setVariableSubArea(subArea);
  }

  private void manageVariableValues(
      List<VariableValue> values, String oldVariableType, String newVariableType) {

    // Exit if it is a variable creation
    if (oldVariableType == null) return;

    // If changing to a type of variable that does not accept variable values
    if (NO_VALUE_VARIABLE_TYPES.contains(newVariableType)) {
      values.clear();
      return;
    }

    // If changing from a constant to another type
    if (CONSTANT.contains(oldVariableType) && !CONSTANT.contains(newVariableType)) {
      values.clear();
      return;
    }

    // If changing from another type to constant
    if (!CONSTANT.contains(oldVariableType) && CONSTANT.contains(newVariableType)) {
      values.clear();
    }
  }

  public String generateHistoricalRange(DataPeriod dataPeriod) {
    String returnRange;

    if (this.getAttachedModel().getNbHistoricalPeriod() > 0) {

      DataPeriod startDataPeriod =
          new DataPeriod(
              this.attachedModel,
              -this.attachedModel.getNbHistoricalPeriod(),
              dataPeriod.segmentOffsetFactor(),
              this.isSingleOrConstantValue());

      String startRange = CellValueHelper.generateAddress(this, startDataPeriod);
      DataPeriod endDataPeriod =
          new DataPeriod(
              this.attachedModel,
              -1,
              dataPeriod.segmentOffsetFactor(),
              this.isSingleOrConstantValue());

      String stopRange = CellValueHelper.generateAddress(this, endDataPeriod);
      returnRange = startRange + ":" + stopRange;
    } else {
      returnRange = "No historical periods";
    }

    return returnRange;
  }

  public String generateProjectionRange(DataPeriod dataPeriod) {
    String returnRange;

    if (this.getAttachedModel().getNbProjectionPeriod() > 0) {
      DataPeriod startDataPeriod =
          new DataPeriod(
              this.attachedModel,
              0,
              dataPeriod.segmentOffsetFactor(),
              this.isSingleOrConstantValue());

      String startRange = CellValueHelper.generateAddress(this, startDataPeriod);

      DataPeriod endDataPeriod =
          new DataPeriod(
              this.attachedModel,
              this.attachedModel.getNbProjectionPeriod() - 1,
              dataPeriod.segmentOffsetFactor(),
              this.isSingleOrConstantValue());

      String stopRange = CellValueHelper.generateAddress(this, endDataPeriod);
      returnRange = startRange + ":" + stopRange;
    } else {
      returnRange = "No projection periods";
    }

    return returnRange;
  }

  public String generateAllPeriodsRange(DataPeriod dataPeriod) {
    String returnRange;
    int backwardsOffset = 0;
    if (this.getAttachedModel().getNbProjectionPeriod() > 0) {

      if (this.getAttachedModel().getNbHistoricalPeriod() > 0) {
        backwardsOffset = -this.getAttachedModel().getNbHistoricalPeriod();
      }
      DataPeriod startDataPeriod =
          new DataPeriod(
              this.attachedModel,
              backwardsOffset,
              dataPeriod.segmentOffsetFactor(),
              this.isSingleOrConstantValue());

      String startRange = CellValueHelper.generateAddress(this, startDataPeriod);

      DataPeriod endDataPeriod =
          new DataPeriod(
              this.attachedModel,
              this.attachedModel.getNbProjectionPeriod() - 1,
              dataPeriod.segmentOffsetFactor(),
              this.isSingleOrConstantValue());

      String stopRange = CellValueHelper.generateAddress(this, endDataPeriod);
      returnRange = startRange + ":" + stopRange;
    } else {
      returnRange = "No projection periods";
    }

    return returnRange;
  }

  public String generateLastPeriodRange(DataPeriod dataPeriod) {
    if (this.getAttachedModel().getNbProjectionPeriod() > 0) {
      DataPeriod lastPeriod =
          new DataPeriod(
              this.attachedModel,
              this.attachedModel.getNbProjectionPeriod() - 1,
              dataPeriod.segmentOffsetFactor(),
              this.isSingleOrConstantValue());

      return CellValueHelper.generateAddress(this, lastPeriod);
    } else {
      return "No projection periods";
    }
  }

  public String generateFirstPeriodRange(DataPeriod dataPeriod) {
    if (this.getAttachedModel().getNbProjectionPeriod() > 0) {
      DataPeriod firstPeriod =
          new DataPeriod(
              this.attachedModel,
              0,
              dataPeriod.segmentOffsetFactor(),
              this.isSingleOrConstantValue());

      return CellValueHelper.generateAddress(this, firstPeriod);
    } else {
      return "No projection periods";
    }
  }

  /**
   * Returns all @VariableValue attached to the specified scenario
   *
   * @param scenario the scenario to be assessed
   * @return A list of @VariableValue
   */
  public List<VariableValue> getVariableValuesForAScenario(SCENARIO scenario) {
    return this.getVariableValues().stream()
        .filter(variableValue -> variableValue.getScenarioNumber() == scenario.ordinal())
        .toList();
  }

  /**
   * Indicates whether the variable is a constant or a single result
   *
   * @return a boolean
   */
  public boolean isSingleOrConstantValue() {
    return CONSTANT_OR_SINGLE.contains(this.getVariableType());
  }

  /**
   * Return the primary column
   *
   * @return the primary column as int
   */
  public int getPrimaryColumn() {
    if (isSingleOrConstantValue()) {
      return this.attachedModel.getConstantColumn();
    } else {
      return this.attachedModel.getFirstProjectionColumn();
    }
  }

  public int getRow() {
    if (attachedModel.getUseAreaAddress()) {
      variableArea=Objects.requireNonNullElseGet(variableArea,
              ()->{
        Area area=new Area(attachedModel,"Stranded area","Stranded area",999);
        attachedModel.getAreas().add(area);
        return area;
      });

      variableSubArea=Objects.requireNonNullElseGet(variableSubArea,()->{
        SubAreaData subAreaData = new SubAreaData(-1L,"Stranded sub-area","Sub area for variables without subarea",false,999,-1L);
        SubArea subArea=new SubArea(subAreaData,variableArea);
        variableArea.getSubAreas().add(subArea);
        return subArea;
      });

      int subAreaIndex =
          2 * (1 + variableArea.getSubAreas().indexOf(variableSubArea)); // Start at 1
      int areaRow =
          attachedModel.getVariables().stream()
              .filter(variable -> variableArea.equals(variable.getVariableArea()))
              .toList()
              .indexOf(this);
      return 1 + subAreaIndex + areaRow;
    } else {
      return Objects.isNull(attachedModel.getVariables())
          ? -1
          : attachedModel.getVariables().indexOf(this);
    }
  }

  public Long getVariableAreaId() {
    return variableArea.getId();
  }

  public Long getVariableSubAreaId() {
    return variableSubArea.getId();
  }

  public boolean isModelledAtSegment() {
    return variableSubArea.isModelledAtSegment();
  }

  @Override
  public VariableData asData() {
    return new VariableData(
        this.id,
        this.getVariableName(),
        this.getVariableDescription(),
        this.getVariableAreaId(),
        this.getVariableSubAreaId(),
        this.getVariableFormula(),
        this.getVariableType(),
        this.getRow(),
        this.getVariableOrder(),
        this.getVariableDepth(),
        this.isVariableApplyToHistoricals(),
        this.getVariableFormat(),
        EntityDTOConverter.asData(this.getVariableValues()),
        this.getVariableDependencies().stream().map(Variable::getId).toList(),
        this.getEvaluatedVariableFormula(),
        this.getOriginalId(),
        "",
        this.historicalValues,
        this.projectionValues,
        this.isModelledAtSegment(),
        this.singleOrConstantValue,
        this.yearToDateValue,
        this.yearToGoValue);
  }

  public boolean isConstant() {
    return CONSTANT.equals(this.getVariableType());
  }

  @Override
  public Long getModelId() {
    return this.getAttachedModel().getId();
  }

  @Override
  public boolean equals(Object obj) {
    if (obj instanceof Variable variable) {
      return Objects.equals(variable.getId(), this.getId());
    }
    return false;
  }

  @Override
  public int hashCode() {
    return getId().hashCode();
  }

  /**
   * Calculates column index based on segment
   *
   * @param segmentIndex zero based segment index, -1 means aggregation
   * @param period the period index relative to the startYear. If this variable is a single_result
   *     or constant the period sent in is disregarded
   * @return the column for the period and segment
   */
  public int columnOfSegmentAndPeriod(int segmentIndex, Integer period) {

    if (this.isSingleOrConstantValue()) {
      period = 0;
    }

    if (period == XLInstance.YTD_PERIOD) period = this.getAttachedModel().getNbProjectionPeriod();

    return new DataPeriod(
            this.attachedModel, period, segmentIndex + 1, this.isSingleOrConstantValue())
        .getColumn();
  }

  /**
   * Calculates column index based on segment
   *
   * @param segment The segment to be checked
   * @param period the period index relative to the startYear
   * @return the column for the period and segment
   */
  public int columnOfSegmentAndPeriod(Segment segment, Integer period) {
    int segmentIndex = -1;
    if (segment != null) {
      segmentIndex = this.getAttachedModel().getSegments().indexOf(segment);
    }
    return this.columnOfSegmentAndPeriod(segmentIndex, period);
  }

  public boolean isPercentOrKPI() {
    return this.getVariableFormat().equals("percent") || this.getVariableType().equals("kpi");
  }

  public boolean isTypeTotal() {
    return NO_VALUE_VARIABLE_TYPES.contains(this.getVariableType());
  }
}
