package app.valuationcontrol.webservice.model.area;

import static app.valuationcontrol.webservice.helpers.ModelChecker.inSameModel;
import static java.lang.String.format;
import static java.util.Arrays.asList;
import static org.springframework.http.ResponseEntity.badRequest;
import static org.springframework.http.ResponseEntity.ok;

import app.valuationcontrol.webservice.EntityService;
import app.valuationcontrol.webservice.model.Model;
import app.valuationcontrol.webservice.model.events.Event;
import app.valuationcontrol.webservice.model.events.Events;
import app.valuationcontrol.webservice.model.events.listeners.AuditLog;
import jakarta.transaction.Transactional;
import jakarta.validation.Valid;
import java.security.Principal;
import org.springframework.data.util.Pair;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;

@RestController
@Transactional
public class AreaController {

  private static final String SWAPPED_ORDER = "Changed order of zone %s and %s";

  private final AuditLog auditlog;
  private final EntityService entityService;
  private final Events events;

  /** Initializes the Area controller and establish link to database */
  public AreaController(AuditLog auditlog, EntityService entityService, Events events) {
    this.auditlog = auditlog;
    this.entityService = entityService;
    this.events = events;
  }

  @PostMapping(value = "/api/model/{modelId}/area")
  @PreAuthorize("authentication.principal.hasModelRole(#existingModel,'EDITOR')")
  public ResponseEntity<Long> createArea(
      @PathVariable("modelId") Model existingModel,
      @Valid @RequestBody AreaData area,
      Principal principal) {

    Area newArea = new Area(area, existingModel);

    existingModel.getAreas().add(newArea);

    entityService.safeCreate(Area.class, newArea);
    Event<Area> areaEvent = Event.created(this, newArea, principal, Area.class, existingModel);
    events.publishCustomEvent(areaEvent);
    events.processEvents(principal);

    return new ResponseEntity<>(newArea.getId(), HttpStatus.CREATED);
  }

  @PostMapping("/api/model/{modelId}/swap/{areaId}/{newAreaId}")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'EDITOR')")
  public ResponseEntity<String> swapAreaOrder(
      @PathVariable(value = "modelId") Model model,
      @PathVariable(value = "areaId") Area area,
      @PathVariable(value = "newAreaId") Area newArea,
      Principal principal) {

    Area oldArea = new Area(area);
    Area oldSecondArea = new Area(newArea);

    if (!inSameModel(model, area, newArea)) {
      return ResponseEntity.badRequest().build();
    }

    int areaOrderBefore = area.getAreaOrder();

    area.setAreaOrder(newArea.getAreaOrder());
    newArea.setAreaOrder(areaOrderBefore);

    entityService.updateAll(
        Area.class, asList(Pair.of(oldArea, area), Pair.of(oldSecondArea, newArea)));

    auditlog.log(
        area.getAttachedModel(),
        format(SWAPPED_ORDER, area.getAreaName(), newArea.getAreaName()),
        principal);

    return ok().build();
  }

  @PutMapping("api/model/{modelId}/area/{areaId}")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'EDITOR')")
  public ResponseEntity<String> updateArea(
      @PathVariable("modelId") Model model,
      @PathVariable("areaId") Area existingArea,
      @Valid @RequestBody AreaData areaData,
      Principal principal) {

    Area oldArea = new Area(existingArea);
    existingArea.updateFrom(areaData);

    if (!inSameModel(model, existingArea)) {
      return ResponseEntity.badRequest().build();
    }

    Event<Area> areaEvent =
        Event.updated(this, oldArea, existingArea, principal, Area.class, model);
    events.publishCustomEvent(areaEvent);
    events.processEvents(principal);

    return ResponseEntity.ok().build();
  }

  @DeleteMapping("/api/model/{modelId}/area/{areaId}")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'EDITOR')")
  public ResponseEntity<String> deleteArea(
      @PathVariable(value = "modelId") Model model,
      @PathVariable(value = "areaId") Area area,
      Principal principal) {

    if (!inSameModel(model, area)) {
      return ResponseEntity.badRequest().build();
    }

    if (!area.getSubAreas().isEmpty()) {
      return badRequest()
          .body(
              "This area cannot be deleted as there are "
                  + area.getSubAreas().size()
                  + " subareas in this area. Please remove them prior to removing this area.");
    }

    if (model.getVariables().stream().anyMatch(v -> v.getVariableAreaId().equals(area.getId()))) {
      return badRequest()
          .body(
              "This area cannot be deleted as there are "
                  + model.getVariables().stream()
                      .filter(v -> v.getVariableAreaId().equals(area.getId()))
                      .count()
                  + " variables in this area. Please remove them prior to removing this area.");
    }

    if (!model.getAreas().contains(area)) {
      return badRequest().body("This area cannot be found in this model");
    }
    model.getAreas().remove(area);
    Event<Area> areaEvent = Event.deleted(this, area, principal, Area.class, model);
    events.publishCustomEvent(areaEvent);
    events.processEvents(principal);

    return ok("Area was deleted correctly");
  }
}
