package app.valuationcontrol.webservice.model.graph;

import static app.valuationcontrol.webservice.helpers.ModelChecker.inSameModel;

import app.valuationcontrol.webservice.EntityService;
import app.valuationcontrol.webservice.model.Model;
import app.valuationcontrol.webservice.model.events.Event;
import app.valuationcontrol.webservice.model.events.Events;
import jakarta.validation.Valid;
import java.security.Principal;
import org.springframework.beans.propertyeditors.CustomNumberEditor;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;

@RestController
public class ModelGraphController {

  private final EntityService entityService;
  private final Events events;

  public ModelGraphController(EntityService entityService, Events events) {
    this.entityService = entityService;
    this.events = events;
  }

  @InitBinder
  public void initBinder(WebDataBinder binder) {
    CustomNumberEditor cne = new CustomNumberEditor(Long.class, true);
    cne.setValue(-1);
    binder.registerCustomEditor(Long.class, cne);
  }

  @PostMapping("/api/model/{modelId}/graph")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'EDITOR')")
  public ResponseEntity<Long> addGraph(
      @PathVariable(value = "modelId") Model model,
      @RequestBody @Valid ModelGraphData graphData,
      Principal principal) {

    ModelGraph newModelGraph = new ModelGraph(graphData, model);

    return entityService
        .safeCreate(ModelGraph.class, newModelGraph)
        .map(
            modelGraph -> {
              Event<ModelGraph> event =
                  Event.created(this, modelGraph, principal, ModelGraph.class, model);
              events.publishCustomEvent(event);
              events.processEvents(principal);
              return new ResponseEntity<>(modelGraph.getId(), HttpStatus.CREATED);
            })
        .orElse(ResponseEntity.badRequest().build());
  }

  @PutMapping("/api/model/{modelId}/graph/{graphId}")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'EDITOR')")
  public ResponseEntity<Long> updateGraph(
      @PathVariable(value = "modelId") Model model,
      @PathVariable(value = "graphId") ModelGraph existingGraph,
      @RequestBody @Valid ModelGraphData updatedGraphData,
      Principal principal) {

    if (!inSameModel(model, existingGraph)) {
      return ResponseEntity.badRequest().build();
    }

    ModelGraph oldGraph = new ModelGraph(existingGraph);
    existingGraph.updateFrom(updatedGraphData);

    Event<ModelGraph> event =
        Event.updated(this, oldGraph, existingGraph, principal, ModelGraph.class, model);
    events.publishCustomEvent(event);
    events.processEvents(principal);

    return ResponseEntity.ok(existingGraph.getId());
  }

  @DeleteMapping("/api/model/{modelId}/graph/{graphId}")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'EDITOR')")
  public ResponseEntity<String> deleteArea(
      @PathVariable(value = "modelId") Model model,
      @PathVariable(value = "graphId") ModelGraph graph,
      Principal principal) {

    if (!inSameModel(model, graph)) {
      return ResponseEntity.badRequest().build();
    }

    model.getGraphs().remove(graph);

    Event<ModelGraph> event = Event.deleted(this, graph, principal, ModelGraph.class, model);
    events.publishCustomEvent(event);
    events.processEvents(principal);

    return ResponseEntity.ok().build();
  }
}
