package app.valuationcontrol.webservice.model.sensitivity;

import app.valuationcontrol.webservice.helpers.DataTransformer;
import app.valuationcontrol.webservice.helpers.ModelProvider;
import app.valuationcontrol.webservice.model.Model;
import jakarta.persistence.Entity;
import jakarta.persistence.GeneratedValue;
import jakarta.persistence.GenerationType;
import jakarta.persistence.Id;
import jakarta.persistence.ManyToOne;
import java.time.LocalDateTime;
import lombok.Getter;
import lombok.Setter;

@Getter
@Entity
public class Sensitivity implements DataTransformer<SensitivityData>, ModelProvider {
  /** */
  @Id
  @GeneratedValue(strategy = GenerationType.IDENTITY)
  @Setter
  private Long id;

  @Setter @ManyToOne private Model attachedModel;

  private String sensitivityName;
  private String sensitivityDescription;
  @Setter private Long sensitivityVariable1Id;
  @Setter private Long sensitivityVariable2Id;
  @Setter private Long sensitivityMeasurementVariableId;

  private Integer sensitivityVariable1Period;
  private Integer sensitivityVariable2Period;
  private Integer sensitivityMeasurementVariablePeriod;

  private Integer sensitivityVariable1Steps;
  private Integer sensitivityVariable2Steps;

  private Float sensitivityVariable1StepSize;
  private Float sensitivityVariable2StepSize;

  @Setter private LocalDateTime sensitivityLastRun;

  public Sensitivity(Sensitivity sensitivity) {
    this(sensitivity.asData(), sensitivity.getAttachedModel());
    this.id = sensitivity.id;
  }

  public Sensitivity(SensitivityData sensitivityData, Model model) {
    this.attachedModel = model;
    this.updateWith(sensitivityData);
  }

  /**
   * Returns relative index for this year compared to models start year
   *
   * @param year e.g. 2022
   * @return relative index from start year
   */
  public static Integer fromYear(Integer year, Model model) {

    if (year == null) {
      return null;
    }

    var yearIndex = year - model.getStartYear();

    if (yearIndex + model.getNbHistoricalPeriod() < 0
        || yearIndex - model.getNbProjectionPeriod() > 0) {
      throw new IllegalArgumentException(
          "Year: "
              + year
              + " is outside of the model ("
              + (model.getStartYear() - model.getNbHistoricalPeriod())
              + " - "
              + (model.getStartYear() + model.getNbProjectionPeriod())
              + ")");
    }

    return yearIndex;
  }

  public Sensitivity() {}

  @Override
  public SensitivityData asData() {
    return new SensitivityData(
        this.id,
        sensitivityName,
        sensitivityDescription,
        sensitivityVariable1Id,
        sensitivityVariable2Id,
        sensitivityMeasurementVariableId,
        asModelYear(sensitivityVariable1Period, getAttachedModel()),
        asModelYear(sensitivityVariable2Period, getAttachedModel()),
        asModelYear(sensitivityMeasurementVariablePeriod, getAttachedModel()),
        sensitivityVariable1Steps,
        sensitivityVariable2Steps,
        sensitivityVariable1StepSize,
        sensitivityVariable2StepSize,
        sensitivityLastRun);
  }

  public static Integer asModelYear(Integer indexedPeriod, Model model) {
    if (indexedPeriod != null && indexedPeriod > model.getNbProjectionPeriod()) {
      return indexedPeriod;
    }
    return asModelYear(indexedPeriod, model.getStartYear());
  }

  public static Integer asModelYear(Integer indexedPeriod, Integer startYear) {
    if (indexedPeriod == null) {
      return null;
    }
    return startYear + indexedPeriod;
  }

  @Override
  public Long getModelId() {
    return this.getAttachedModel().getId();
  }

  public void updateWith(SensitivityData sensitivityData) {
    this.sensitivityName = sensitivityData.name();
    this.sensitivityDescription = sensitivityData.description();

    this.sensitivityVariable1Id = sensitivityData.variable1Id();
    this.sensitivityVariable2Id = sensitivityData.variable2Id();
    this.sensitivityMeasurementVariableId = sensitivityData.measurementVariableId();
    this.sensitivityVariable1Period =
        fromYear(sensitivityData.variable1Period(), this.attachedModel);
    this.sensitivityVariable2Period =
        fromYear(sensitivityData.variable2Period(), this.attachedModel);
    this.sensitivityMeasurementVariablePeriod =
        fromYear(sensitivityData.measurementVariablePeriod(), this.attachedModel);
    this.sensitivityVariable1Steps = sensitivityData.variable1Steps();
    this.sensitivityVariable2Steps = sensitivityData.variable2Steps();
    this.sensitivityVariable1StepSize = sensitivityData.variable1StepSize();
    this.sensitivityVariable2StepSize = sensitivityData.variable2StepSize();
  }
}
