package app.valuationcontrol.webservice.model.subarea;

import static app.valuationcontrol.webservice.helpers.ModelChecker.inSameModel;
import static java.lang.String.format;
import static org.springframework.http.ResponseEntity.ok;

import app.valuationcontrol.webservice.EntityService;
import app.valuationcontrol.webservice.model.Model;
import app.valuationcontrol.webservice.model.area.Area;
import app.valuationcontrol.webservice.model.events.Event;
import app.valuationcontrol.webservice.model.events.Events;
import app.valuationcontrol.webservice.model.events.listeners.AuditLog;
import jakarta.validation.Valid;
import java.security.Principal;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;

@RestController
public class SubAreaController {

  private static final String SWAPPED_SUB_AREA_ORDER = "Change order of %s and %s";
  private final EntityService entityService;
  private final AuditLog auditLog;

  private final Events events;

  /** Initializes the Area controller and establish link to database */
  public SubAreaController(EntityService entityService, AuditLog auditLog, Events events) {
    this.entityService = entityService;
    this.events = events;
    this.auditLog = auditLog;
  }

  @PostMapping("/api/model/{modelId}/area/{areaId}/subarea")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'EDITOR')")
  public ResponseEntity<Long> addSubArea(
      @PathVariable(value = "modelId") Model model,
      @PathVariable(value = "areaId") Area area,
      @Valid @RequestBody SubAreaData subAreaData,
      Principal principal) {

    SubArea subArea = new SubArea(subAreaData, area);
    area.getSubAreas().add(subArea);

    return entityService
        .safeCreate(SubArea.class, subArea, model, area)
        .map(
            createdSubArea -> {
              Event<SubArea> event =
                  Event.created(this, createdSubArea, principal, SubArea.class, model);
              events.publishCustomEvent(event);
              events.processEvents(principal);
              return new ResponseEntity<>(createdSubArea.getId(), HttpStatus.CREATED);
            })
        .orElse(ResponseEntity.badRequest().build());
  }

  @PutMapping("/api/model/{modelId}/area/{areaId}/subarea/{subAreaId}")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'EDITOR')")
  public ResponseEntity<String> updateSubArea(
      @PathVariable(value = "modelId") Model model,
      @PathVariable(value = "areaId") Area area,
      @PathVariable(value = "subAreaId") SubArea subArea,
      @Valid @RequestBody SubAreaData subAreaData,
      Principal principal) {

    if (!inSameModel(model, area, subArea)) {
      return ResponseEntity.badRequest().build();
    }
    SubArea oldSubArea = new SubArea(subArea);

    subArea.updateWith(subAreaData, area);

    Event<SubArea> event =
        Event.updated(this, oldSubArea, subArea, principal, SubArea.class, model);
    events.publishCustomEvent(event);
    events.processEvents(principal);
    return ResponseEntity.ok(String.valueOf(subArea.getId()));
  }

  @PostMapping("/api/model/{modelId}/area/{areaId}/swap/{subAreaId1}/{subAreaId2}")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'EDITOR')")
  public ResponseEntity<String> swapSubAreas(
      @PathVariable(value = "modelId") Model model,
      @PathVariable(value = "areaId") Area area,
      @PathVariable(value = "subAreaId1") SubArea subArea1,
      @PathVariable(value = "subAreaId2") SubArea subArea2,
      Principal principal) {

    if (!inSameModel(model, area, subArea1, subArea2)) {
      return ResponseEntity.badRequest().build();
    }

    int subArea1OrderBefore = subArea1.getSubAreaOrder();
    subArea1.setSubAreaOrder(subArea2.getSubAreaOrder());
    subArea2.setSubAreaOrder(subArea1OrderBefore);

    Event<Model> event = Event.lightUpdated(this, model, principal, Model.class, model);
    events.publishCustomEvent(event);
    events.processEvents(principal);

    auditLog.log(
        area.getAttachedModel(),
        format(SWAPPED_SUB_AREA_ORDER, subArea1.getSubAreaName(), subArea2.getSubAreaName()),
        principal);

    return ok().build();
  }

  @DeleteMapping("/api/model/{modelId}/area/{areaId}/subarea/{subAreaId}")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'EDITOR')")
  public ResponseEntity<String> deleteArea(
      @PathVariable(value = "modelId") Model model,
      @PathVariable(value = "areaId") Area area,
      @PathVariable(value = "subAreaId") SubArea subArea,
      Principal principal) {

    if (!inSameModel(model, area, subArea)) {
      return ResponseEntity.badRequest().build();
    }

    var numberOfVariablesInSubArea =
        model.getVariables().stream()
            .filter(variable -> variable.getVariableSubArea().getId() == subArea.getId())
            .count();

    if (numberOfVariablesInSubArea > 0) {
      return ResponseEntity.badRequest()
          .body(
              "Cannot delete subarea with "
                  + numberOfVariablesInSubArea
                  + " variables still connected to it");
    } else {
      area.getSubAreas().remove(subArea);
      Event<SubArea> event = Event.deleted(this, subArea, principal, SubArea.class, model);
      events.publishCustomEvent(event);
      events.processEvents(principal);
    }

    return ok().build();
  }
}
