package app.valuationcontrol.webservice.model.variable;

import static app.valuationcontrol.webservice.helpers.ModelChecker.inSameModel;
import static app.valuationcontrol.webservice.helpers.ModelChecker.isProtectedName;
import static app.valuationcontrol.webservice.model.ModelController.MODEL_ID;
import static app.valuationcontrol.webservice.model.events.Event.deleted;
import static java.util.Arrays.asList;
import static org.springframework.http.ResponseEntity.ok;

import app.valuationcontrol.webservice.EntityService;
import app.valuationcontrol.webservice.helpers.exceptions.ResourceException;
import app.valuationcontrol.webservice.model.Model;
import app.valuationcontrol.webservice.model.area.Area;
import app.valuationcontrol.webservice.model.events.Event;
import app.valuationcontrol.webservice.model.events.Events;
import app.valuationcontrol.webservice.model.subarea.SubArea;
import app.valuationcontrol.webservice.model.variablevalue.ImportVariableValueData;
import app.valuationcontrol.webservice.model.variablevalue.VariableValue;
import app.valuationcontrol.webservice.model.variablevalue.VariableValueData;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import jakarta.validation.Valid;
import java.security.Principal;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import lombok.extern.log4j.Log4j2;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;

@RestController
@Log4j2
@Transactional
public class VariableController {

  private final EntityService entityService;
  private final Events events;

  public static final String MODEL_ID_DESCRIPTION = "The id of the model containing the variable";
  public static final String VARIABLE_ID = "variableId";
  public static final String VARIABLE_ID_DESCRIPTION = "The id of the variable to be amended";

  public static final String VARIABLE_VALUE_ID = "variableValueId";
  public static final String VARIABLE_VALUE_ID_DESCRIPTION = "The id of the value to be amended";

  public static final String AREA_ID = "areaId";
  public static final String AREA_ID_DESCRIPTION = "The id of the area containing the variable";

  public static final String SUB_AREA_ID = "subAreaId";
  public static final String SUB_AREA_ID_DESCRIPTION =
      "The id of the sub-area containing the variable";

  /**
   * Initializes the model controller and establish link to database Creator function that ensures
   * that the model is connected to a document
   */
  public VariableController(EntityService entityService, Events events) {
    this.entityService = entityService;
    this.events = events;
  }

  @Operation(
      summary = "Swap order of two variables",
      description = "Use this entrypoint to swap the order of two variable in the model",
      responses = {
        @ApiResponse(responseCode = "200", description = "Successfull operation"),
        @ApiResponse(responseCode = "400", description = "Invalid request parameters"),
        @ApiResponse(responseCode = "401", description = "Unauthorized access"),
        @ApiResponse(responseCode = "500", description = "Server error"),
      })
  @PostMapping("/api/model/{modelId}/swapvariable/{variable1}/{variable2}")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'EDITOR')")
  public ResponseEntity<String> swapVariableOrder(
      @Parameter(description = MODEL_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = MODEL_ID)
          Model model,
      @Parameter(
              description = "The id of the first variable to be swapped",
              in = ParameterIn.PATH,
              required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = "variable1")
          Variable variable1,
      @Parameter(
              description = "The id of the second variable to be swapped",
              in = ParameterIn.PATH,
              required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = "variable2")
          Variable variable2,
      Principal principal) {

    if (inSameModel(variable1, variable2)) {

      Integer variable1Order = variable1.getVariableOrder();
      variable1.setVariableOrder(variable2.getVariableOrder());
      variable2.setVariableOrder(variable1Order);

      Event<Model> modelEvent = Event.lightUpdated(this, model, principal, Model.class, model);
      events.publishCustomEvent(modelEvent);
      events.processEvents(principal);

      return ok().build();
    } else {
      throw new ResourceException(HttpStatus.BAD_REQUEST, "Variables are not in the same model");
    }
  }

  /**
   * Adds a variable to the @model
   *
   * @param model the model to add the variable to
   * @param variableData the data to be stored
   * @param principal the user logged in
   * @return the id of the created variable
   */
  @Operation(
      summary = "Create a new variable in the model",
      description = "Use this entrypoint to add a variable to the model",
      responses = {
        @ApiResponse(responseCode = "201", description = "the id of the created variable"),
        @ApiResponse(responseCode = "400", description = "Invalid request parameters"),
        @ApiResponse(responseCode = "401", description = "Unauthorized access"),
        @ApiResponse(responseCode = "500", description = "Server error"),
      })
  @PostMapping("/api/model/{modelId}/area/{areaId}/subarea/{subAreaId}/variable")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'EDITOR')")
  public ResponseEntity<Long> createVariable(
      @Parameter(description = MODEL_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = MODEL_ID)
          Model model,
      @Parameter(description = AREA_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = AREA_ID)
          Area area,
      @Parameter(description = SUB_AREA_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = SUB_AREA_ID)
          SubArea subArea,
      @RequestBody @Valid VariableData variableData,
      Principal principal) {

    if (!inSameModel(model, area, subArea)) {
      throw new ResourceException(HttpStatus.BAD_REQUEST, "Check the consistency of the request");
    }
    if (isProtectedName(variableData.variableName())) {
      throw new ResourceException(
          HttpStatus.BAD_REQUEST,
          variableData.variableName()
              + " is a protected name (Excel function), please change the name of the variable");
    }
    Variable variable = new Variable(variableData, model, area, subArea);
    model.getVariables().add(variable);

    return entityService
        .safeCreate(Variable.class, variable, model, area, subArea)
        .map(
            createVariable -> {
              Event<Variable> variableEvent =
                  Event.created(this, createVariable, principal, Variable.class, model);
              events.publishCustomEvent(variableEvent);
              events.processEvents(principal);
              return new ResponseEntity<>(createVariable.getId(), HttpStatus.CREATED);
            })
        .orElse(ResponseEntity.badRequest().build());
  }

  /**
   * Update a variable to the @model
   *
   * @param model the model to add the variable to
   * @param variableData the data to be stored
   * @param principal the user logged in
   * @return the id of the created variable
   */
  @Operation(
      summary = "Update a variable",
      description = "Use this entrypoint to update a variable to the model",
      responses = {
        @ApiResponse(responseCode = "200", description = "Successfull operation"),
        @ApiResponse(responseCode = "400", description = "Invalid request parameters"),
        @ApiResponse(responseCode = "401", description = "Unauthorized access"),
        @ApiResponse(responseCode = "500", description = "Server error"),
      })
  @PutMapping("/api/model/{modelId}/area/{areaId}/subarea/{subAreaId}/variable/{variableId}")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'EDITOR')")
  public ResponseEntity<String> updateVariable(
      @Parameter(description = MODEL_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = MODEL_ID)
          Model model,
      @Parameter(description = AREA_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = AREA_ID)
          Area area,
      @Parameter(description = SUB_AREA_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = SUB_AREA_ID)
          SubArea subArea,
      @Parameter(description = VARIABLE_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(VARIABLE_ID)
          Variable existingVariable,
      @RequestBody @Valid VariableData variableData,
      Principal principal) {

    if (!inSameModel(model, area, subArea, existingVariable)) {
      return ResponseEntity.badRequest().build();
    }

    if (isProtectedName(variableData.variableName())) {
      throw new ResourceException(
          HttpStatus.BAD_REQUEST,
          variableData.variableName()
              + " is a protected name, please change the name of the variable");
    }
    // Creates a detached version of the existing variable.
    // Detached means that it is not connected to the variableRepository, using new to detach
    Variable oldVariable = new Variable(existingVariable);

    // Update key fields all apart id,model and order (not possible to change id nor model),
    // order is changed through swapOrder
    existingVariable.updateFromVariableData(variableData, area, subArea);
    // existingVariable.getAttachedModel().getVariables()

    Event<Variable> variableEvent =
        Event.updated(this, oldVariable, existingVariable, principal, Variable.class, model);
    events.publishCustomEvent(variableEvent);
    events.processEvents(principal);

    return ResponseEntity.ok().build();
  }

  /**
   * Safely deletes a variable from the @model
   *
   * @param model the model to add the variable to
   * @param area the area the variable is connected to
   * @param subArea the subArea the variable is connected to
   * @return the id of the created variable
   */
  @Operation(
      summary = "Delete a variable",
      description = "Use this entrypoint to delete a variable to the model",
      responses = {
        @ApiResponse(responseCode = "200", description = "Successfull operation"),
        @ApiResponse(responseCode = "400", description = "Invalid request parameters"),
        @ApiResponse(responseCode = "401", description = "Unauthorized access"),
        @ApiResponse(responseCode = "500", description = "Server error"),
      })
  @DeleteMapping("/api/model/{modelId}/area/{areaId}/subarea/{subAreaId}/variable/{variableId}")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'EDITOR')")
  public ResponseEntity<String> deleteVariable(
      @Parameter(description = MODEL_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = MODEL_ID)
          Model model,
      @Parameter(description = AREA_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = AREA_ID)
          Area area,
      @Parameter(description = SUB_AREA_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = SUB_AREA_ID)
          SubArea subArea,
      @PathVariable("variableId") Variable existingVariable,
      Principal principal) {
    if (!inSameModel(model, area, subArea, existingVariable)) {
      return ResponseEntity.status(HttpStatus.FORBIDDEN).build();
    }

    if (usedInSensitivity(model, existingVariable)) {
      return ResponseEntity.badRequest()
          .body("Cannot delete a variable used in a Sensitivity, please edit Sensitivity first");
    }

    if (usedInGraphs(model, existingVariable)) {
      return ResponseEntity.badRequest()
          .body("Cannot delete a variable used in a Graph, please edit Graphs first");
    }

    if (usedInOtherCalculations(existingVariable)) {
      return ResponseEntity.badRequest()
          .body("Cannot delete a variable used in another variables calculation");
    }

    // Delete variable from keyParam if it is there
    if (model.getKeyParam() != null) {
      model.getKeyParam().deleteVariableFromKeyParam(existingVariable.getId());
    }

    existingVariable.getVariableDependencies().clear();
    model.getVariables().remove(existingVariable);

    Event<Variable> event = deleted(this, existingVariable, principal, Variable.class, model);
    events.publishCustomEvent(event);
    events.processEvents(principal);

    return ResponseEntity.ok().build();
  }

  private void updateVariableValueWithEvent(
      Model model,
      VariableValue variableValue,
      Variable variable,
      VariableValueData variableValueData,
      Principal principal) {
    VariableValue variableValueBefore = new VariableValue(variableValue.asData(), variable);
    variableValue.setValue(variableValueData.value());
    variableValue.setEditor(principal.getName());
    if (variableValue.getSourceFile() == null || variableValue.getSourceFile().isEmpty())
      variableValue.setSourceFile("Manually registered");

    if (variable.isConstant()) {
      variableValue.setPeriod(null);
    }

    if (!variable.isModelledAtSegment()) {
      variableValue.setAttachedSegment(null);
    }

    Event<VariableValue> event =
        Event.updated(
            this, variableValueBefore, variableValue, principal, VariableValue.class, model);
    events.publishCustomEvent(event);
  }

  private VariableValue createAndSaveVariableValueWithEvent(
      Model model, Variable variable, VariableValueData variableValueData, Principal principal) {

    // Detecting duplicates
    log.debug("Detecting duplicates");
    try {
      if (variable.getVariableValues().stream()
          .anyMatch(
              variableValue -> {
                boolean isSamePeriod =
                    Objects.equals(variableValue.getPeriod(), variableValueData.period());
                boolean isSameScenario =
                    Objects.equals(
                        variableValue.getScenarioNumber(), variableValueData.scenarioNumber());
                boolean isSameNullSegment =
                    variableValue.getAttachedSegment() == null
                        && (variableValueData.attachedSegmentId() == null
                            || variableValueData.attachedSegmentId() <= 0);
                boolean isSameNonNullSegment =
                    variableValue.getAttachedSegment() != null
                        && variableValueData.attachedSegmentId() != null
                        && (variableValue.getAttachedSegment().getId()
                            == variableValueData.attachedSegmentId());
                return isSamePeriod
                    && isSameScenario
                    && (isSameNullSegment || isSameNonNullSegment);
              })) {
        log.info("Duplicate variable data is detected" + variableValueData.value());
        throw new ResourceException(
            HttpStatus.BAD_REQUEST, "Duplicate variable value data is detected");
      }
    } catch (Exception e) {
      log.error(e);
      throw new ResourceException(
          HttpStatus.BAD_REQUEST, "Duplicate variable value data is detected");
    }

    log.debug("Checking segment");
    if (variable.isModelledAtSegment() && variableValueData.attachedSegmentId() == null) {
      log.info(
          "Received variable value for a variable modelled at segment but no segment was attached "
              + variableValueData.value());
      throw new ResourceException(
          HttpStatus.BAD_REQUEST, "Trying to set a value on a variable defined at segment");
    }

    if (variableValueData.attachedSegmentId() != null
        && variableValueData.attachedSegmentId() > 0
        && model.getSegments().stream()
            .noneMatch(s -> s.getId() == variableValueData.attachedSegmentId())) {
      log.info(
          "AttachedSegmentId was not found in model segments"
              + variableValueData.attachedSegmentId());
      throw new ResourceException(
          HttpStatus.BAD_REQUEST, "Trying to set a value on a variable defined at segment");
    }

    log.debug("Checking variable");
    if (variableValueData.attachedVariableId() != null
        && (variableValueData.attachedVariableId().equals(-1L) // Allowing -1 as variableID
            || !variableValueData.attachedVariableId().equals(variable.getId()))) {
      log.info("Invalid attached variableID");
      throw new ResourceException(
          HttpStatus.BAD_REQUEST, "VariableValueData had no valid variable id ");
    }

    log.debug("Converting variableValueData to variableValue");

    VariableValue variableValue = new VariableValue(variableValueData, variable);
    variableValue.setEditor(principal.getName());
    if (variableValue.getSourceFile() == null || variableValue.getSourceFile().isEmpty())
      variableValue.setSourceFile("Manually registered");

    if (variable.isConstant()) {
      variableValue.setPeriod(null);
    }

    log.debug("Checking modelled at segment");

    if (!variable.isModelledAtSegment()) {
      variableValue.setAttachedSegment(null);
    } else if (variableValueData.attachedSegmentId() != null
        && variableValueData.attachedSegmentId() > 0) {
      model.getSegments().stream()
          .filter(s -> Long.valueOf(s.getId()).equals(variableValueData.attachedSegmentId()))
          .forEach(variableValue::setAttachedSegment);
    }

    try {
      variable.getVariableValues().add(variableValue);

      variableValue =
          entityService.safeCreate(VariableValue.class, variableValue, variable).orElseThrow();
    } catch (Exception e) {
      log.error(e);
    }

    Event<VariableValue> event =
        Event.created(this, variableValue, principal, VariableValue.class, model);
    events.publishCustomEvent(event);
    return variableValue;
  }

  @Operation(
      summary = "Add a single value to a variable",
      description =
          "Use this entrypoint to add a single value to a variable. The value is valid for a defined scenario and period",
      responses = {
        @ApiResponse(responseCode = "201", description = "The id of the created variable value"),
        @ApiResponse(responseCode = "400", description = "Invalid request parameters"),
        @ApiResponse(responseCode = "401", description = "Unauthorized access"),
        @ApiResponse(responseCode = "500", description = "Server error"),
      })
  @PostMapping("/api/model/{modelId}/variable/{variableId}/value")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'EDITOR')")
  public ResponseEntity<Long> addVariableValue(
      @Parameter(description = MODEL_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = MODEL_ID)
          Model model,
      @Parameter(description = VARIABLE_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = VARIABLE_ID)
          Variable variable,
      @RequestBody @Valid VariableValueData variableValueData,
      Principal principal) {

    if (!inSameModel(model, variable)) {
      return ResponseEntity.badRequest().build();
    }

    VariableValue variableValue =
        createAndSaveVariableValueWithEvent(model, variable, variableValueData, principal);

    if (variableValue.getId() > 0) {
      events.processEvents(principal);
      return new ResponseEntity<>(variableValue.getId(), HttpStatus.CREATED);
    } else {
      return ResponseEntity.badRequest().build();
    }
  }

  @Operation(
      summary = "Import several values at the same time",
      description = "Use this entrypoint to add several values to several variables.",
      responses = {
        @ApiResponse(responseCode = "201", description = "The ids of the created variable values"),
        @ApiResponse(responseCode = "400", description = "Invalid request parameters"),
        @ApiResponse(responseCode = "401", description = "Unauthorized access"),
        @ApiResponse(responseCode = "500", description = "Server error"),
      })
  @PostMapping("/api/model/{modelId}/variablevalues/import")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'EDITOR')")
  public ResponseEntity<List<Long>> importVariableValues(
      @Parameter(description = MODEL_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = MODEL_ID)
          Model model,
      @RequestBody @Valid ImportVariableValueData importVariableValueData,
      Principal principal) {
    log.info("Creating a new model import from filename: " + importVariableValueData.sourceFile());

    return addOrUpdateVariableValues(model, importVariableValueData.variableValueData(), principal);
  }

  /**
   * Add or update variables values
   *
   * @param model the model to be updated sent by its model id
   * @param variableValueDataList a list of VariableValueData
   * @param principal the principal performing the action
   * @return a list of created variableValueIDs
   */
  @Operation(
      summary = "Create or update several values at the same time",
      description = "Use this entrypoint to create or update several values to several variables.",
      responses = {
        @ApiResponse(responseCode = "201", description = "The ids of the created variable values"),
        @ApiResponse(responseCode = "400", description = "Invalid request parameters"),
        @ApiResponse(responseCode = "401", description = "Unauthorized access"),
        @ApiResponse(responseCode = "500", description = "Server error"),
      })
  @PostMapping("/api/model/{modelId}/variablevalues")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'EDITOR')")
  public ResponseEntity<List<Long>> addOrUpdateVariableValues(
      @Parameter(description = MODEL_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = MODEL_ID)
          Model model,
      @RequestBody @Valid List<VariableValueData> variableValueDataList,
      Principal principal) {

    List<Long> ids = new ArrayList<>();
    for (VariableValueData variableValueData : variableValueDataList) {
      model
          .getVariableWithID(variableValueData.attachedVariableId())
          .ifPresent(
              variable -> {
                if (variableValueData.id() == -1L) {
                  ids.add(
                      this.createAndSaveVariableValueWithEvent(
                              model, variable, variableValueData, principal)
                          .getId());
                } else {
                  variable.getVariableValues().stream()
                      .filter(vv -> Objects.equals(vv.getId(), variableValueData.id()))
                      .findFirst()
                      .ifPresent(
                          variableValueFromDb -> {
                            updateVariableValueWithEvent(
                                model, variableValueFromDb, variable, variableValueData, principal);
                            ids.add(variableValueFromDb.getId());
                          });
                }
              });
    }
    events.processEvents(principal);
    return ResponseEntity.ok(ids);
  }

  @Operation(
      summary = "Update a single variable value",
      description = "Use this entrypoint to update a single variable value",
      responses = {
        @ApiResponse(responseCode = "200", description = "Successfull operation"),
        @ApiResponse(responseCode = "400", description = "Invalid request parameters"),
        @ApiResponse(responseCode = "401", description = "Unauthorized access"),
        @ApiResponse(responseCode = "500", description = "Server error"),
      })
  @PutMapping("/api/model/{modelId}/variable/{variableId}/value/{variableValueId}")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'EDITOR')")
  public ResponseEntity<String> updateVariableValue(
      @Parameter(description = MODEL_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = MODEL_ID)
          Model model,
      @Parameter(description = VARIABLE_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = VARIABLE_ID)
          Variable variable,
      @Parameter(
              description = VARIABLE_VALUE_ID_DESCRIPTION,
              in = ParameterIn.PATH,
              required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = VARIABLE_VALUE_ID)
          VariableValue variableValue,
      @RequestBody @Valid VariableValueData variableValueData,
      Principal principal) {

    if (!inSameModel(model, variable, variableValue)) {
      return ResponseEntity.badRequest().build();
    }

    updateVariableValueWithEvent(
        model, variableValue, variable, variableValueData, principal); // generates event
    events.processEvents(principal);

    return ResponseEntity.ok("");
  }

  @Operation(
      summary = "Delete a single variable value",
      description = "Use this entrypoint to delete a single variable value",
      responses = {
        @ApiResponse(responseCode = "200", description = "Successfull operation"),
        @ApiResponse(responseCode = "400", description = "Invalid request parameters"),
        @ApiResponse(responseCode = "401", description = "Unauthorized access"),
        @ApiResponse(responseCode = "500", description = "Server error"),
      })
  @DeleteMapping("/api/model/{modelId}/variable/{variableId}/value/{variableValueId}")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'EDITOR')")
  public ResponseEntity<String> deleteCellValue(
      @Parameter(description = MODEL_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = MODEL_ID)
          Model model,
      @Parameter(description = VARIABLE_ID_DESCRIPTION, in = ParameterIn.PATH, required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = VARIABLE_ID)
          Variable variable,
      @Parameter(
              description = VARIABLE_VALUE_ID_DESCRIPTION,
              in = ParameterIn.PATH,
              required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = VARIABLE_VALUE_ID)
          VariableValue variableValue,
      Principal principal) {

    if (!inSameModel(model, variable, variableValue)) {
      return ResponseEntity.badRequest().build();
    }

    variable
        .getVariableValues()
        .removeIf(variableValue1 -> variableValue1.getId() == variableValue.getId());

    Event<VariableValue> event =
        Event.deleted(this, variableValue, principal, VariableValue.class, model);
    events.publishCustomEvent(event);
    events.processEvents(principal);

    return ResponseEntity.ok().build();
  }

  private boolean usedInOtherCalculations(Variable existingVariable) {
    return existingVariable.getAttachedModel().getVariables().stream()
        .filter(modelVariable -> !Objects.equals(existingVariable, modelVariable))
        .anyMatch(variable -> variable.getVariableDependencies().contains(existingVariable));
  }

  private boolean usedInGraphs(Model model, Variable existingVariable) {
    return model.getGraphs().stream()
        .anyMatch(
            modelGraph ->
                asList(
                        modelGraph.getGraphVariable1Id(),
                        modelGraph.getGraphVariable2Id(),
                        modelGraph.getGraphVariable3Id())
                    .contains(existingVariable.getId()));
  }

  private boolean usedInSensitivity(Model model, Variable existingVariable) {
    return model.getSensitivities().stream()
        .anyMatch(
            sensitivity ->
                sensitivity.getSensitivityVariable1Id().equals(existingVariable.getId())
                    || sensitivity.getSensitivityVariable2Id().equals(existingVariable.getId()));
  }
}
