package app.valuationcontrol.webservice.user;

import app.valuationcontrol.webservice.EntityService;
import app.valuationcontrol.webservice.helpers.EntityDTOConverter;
import app.valuationcontrol.webservice.helpers.exceptions.ResourceException;
import app.valuationcontrol.webservice.model.Model;
import app.valuationcontrol.webservice.model.ModelController;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import jakarta.validation.Valid;
import java.security.Principal;
import java.util.ArrayList;
import java.util.List;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

@RestController
public class UserController {
  private final UserRepository userRepository;
  private final EntityService entityService;

  /**
   * Initializes the User controller and establish link to database Creator function that ensures
   * that the model is connected to a document
   */
  public UserController(UserRepository userRepository, EntityService entityService) {
    this.userRepository = userRepository;
    this.entityService = entityService;
  }

  @GetMapping("/api/user/models")
  @ResponseBody
  public ResponseEntity<UserData> getModels(Principal principal) {
    ResponseEntity<UserData> myResponseEntity;

    if (principal != null) {
      User user = userRepository.findByEmail(principal.getName());

      if (user != null) {
        myResponseEntity = ResponseEntity.ok(EntityDTOConverter.asData(user));
      } else {
        User user2 = new User();
        user2.setEmail(principal.getName());
        myResponseEntity = ResponseEntity.ok(EntityDTOConverter.asData(user2));
      }
    } else {
      throw new ResourceException(
          HttpStatus.BAD_REQUEST, "You do not have access to any models yet");
    }

    return myResponseEntity;
  }

  @Operation(
      summary = "List all users having access to the model",
      responses = {
        @ApiResponse(responseCode = "200", description = "Successfull operation"),
        @ApiResponse(responseCode = "400", description = "Invalid request parameters"),
        @ApiResponse(responseCode = "401", description = "Unauthorized access"),
        @ApiResponse(responseCode = "500", description = "Server error"),
      })
  @GetMapping("/api/model/{modelId}/users")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'READER')")
  @ResponseBody
  public ResponseEntity<List<UserAccessData>> getUserModelAccess(
      @Parameter(
              description = ModelController.MODEL_ID_DESCRIPTION,
              in = ParameterIn.PATH,
              required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = ModelController.MODEL_ID)
          Model model,
      Principal principal) {

    if (principal == null)
      throw new ResourceException(
          HttpStatus.BAD_REQUEST, "You do not have access to any models yet");

    ArrayList<User> users = userRepository.findByModel(model);

    return ResponseEntity.ok(
        users.stream()
            .map(
                user ->
                    new UserAccessData(
                        user.getId(),
                        user.getEmail(),
                        User.MODEL_ROLE.valueOf(user.getModelRoles().get(model))))
            .toList());
  }

  @PutMapping(value = "/api/model/{modelId}/role/{role}", consumes = "application/json")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'ADMIN')")
  public ResponseEntity<Long> addUserToModel(
      @Parameter(
              description = ModelController.MODEL_ID_DESCRIPTION,
              in = ParameterIn.PATH,
              required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = ModelController.MODEL_ID)
          Model model,
      @PathVariable(value = "role") User.MODEL_ROLE role,
      @Valid @RequestBody Email email,
      Principal principal) {

    User existingUser = userRepository.findByEmail(email.value());

    if (existingUser == null) {
      final User newUser = new User();
      newUser.setEmail(email.value());
      existingUser = entityService.create(User.class, newUser);
    }

    existingUser.addModel(model, role);

    entityService.safeUpdate(User.class, existingUser, existingUser);

    return new ResponseEntity<>(HttpStatus.OK);
  }

  @DeleteMapping("/api/model/{modelId}/user/{userId}")
  @PreAuthorize("authentication.principal.hasModelRole(#model,'ADMIN')")
  public ResponseEntity<Void> removeUserModelRole(
      @Parameter(
              description = ModelController.MODEL_ID_DESCRIPTION,
              in = ParameterIn.PATH,
              required = true)
          @Schema(type = "Integer", minimum = "1")
          @PathVariable(value = ModelController.MODEL_ID)
          Model model,
      @Parameter(description = "The id of the user to be deleted") @PathVariable Long userId,
      Principal principal) {
    final User existingUser = userRepository.getReferenceById(userId);
    final User principalUser = userRepository.findByEmail(principal.getName());

    if (existingUser == null) {
      return ResponseEntity.notFound().build();
    }

    if (existingUser.equals(principalUser)) {
      throw new ResourceException(
          HttpStatus.BAD_REQUEST,
          "You cannot delete yourself from this model, please ask another administrator");
    }

    existingUser.getModelRoles().remove(model);

    entityService.safeUpdate(User.class, existingUser, existingUser);

    return new ResponseEntity<>(HttpStatus.OK);
  }
}
