package app.valuationcontrol.webservice.xlhandler;

import lombok.extern.log4j.Log4j2;
import org.apache.poi.ss.formula.FormulaParseException;
import org.apache.poi.ss.formula.eval.NotImplementedException;
import org.apache.poi.ss.usermodel.CreationHelper;
import org.apache.poi.ss.usermodel.FormulaError;
import org.apache.poi.ss.util.CellUtil;
import org.apache.poi.xssf.usermodel.*;

@Log4j2
public class POISheet implements GenericSheet {

  private final String name;
  private final XSSFSheet sheet;

  public POISheet(String name, XSSFSheet sheet) {
    this.name = name;
    this.sheet = sheet;
  }

  @Override
  public String getName() {
    return this.name;
  }

  @Override
  public void setFormula(String formula, int column, int row) {
    // log.info("Setting formula for " + formula);
    XSSFRow xssfRow = this.sheet.getRow(row);
    if (xssfRow == null) xssfRow = this.sheet.createRow(row);
    XSSFCell cell = (XSSFCell) CellUtil.getCell(xssfRow, column);

    if (formula == null || formula.isBlank()) {
      cell.removeFormula();
      cell.setBlank();
    } else if (formula.startsWith("=")) {
      formula = formula.replaceFirst("=", "");
      formula = formula.replace(";", ",");
      try {
        cell.setCellFormula(formula);
      } catch (FormulaParseException e) {
        log.debug("Couldn't parse " + e + " due to " + e.getMessage());

        cell.removeFormula();
        cell.setCellErrorValue(FormulaError.NA);

        CreationHelper creationHelper = sheet.getWorkbook().getCreationHelper();
        XSSFComment comment =
            sheet.createDrawingPatriarch().createCellComment(creationHelper.createClientAnchor());
        comment.setString(creationHelper.createRichTextString(e.getMessage()));
        comment.setAuthor("Autogenerated comment from ValuationControl");
        cell.setCellComment(comment);
      }
    } else {
      try {
        double d = Double.parseDouble(formula);
        cell.removeFormula();
        cell.setCellValue(d);
      } catch (NumberFormatException nfe) {
        cell.removeFormula();
        cell.setCellValue(formula);
      }
    }
  }

  @Override
  public void setCellValues(CellValue... cellValues) {
    for (CellValue cellValue : cellValues) {
      setFormula(Double.toString(cellValue.value()), cellValue.column(), cellValue.row());
    }
  }

  @Override
  public void computeCellValues(CellValue... cellValues) {
    for (CellValue cellValue : cellValues) {
      computeCell(cellValue.column(), cellValue.column());
    }
  }

  @Override
  public void setValue(String[][] values, int columns, int rows) {
    for (int i = 0; i <= rows; i++) {
      for (int j = 0; j <= columns; j++) {
        setFormula(values[i][j], j, i);
      }
    }
  }

  public XSSFCell customGetCell(int column, int row) {
    XSSFRow xssfRow = this.sheet.getRow(row);
    if (xssfRow == null) xssfRow = this.sheet.createRow(row);
    return (XSSFCell) CellUtil.getCell(xssfRow, column);
  }

  @Override
  public void computeCalculations() {
    // log.debug("Refreshing calculations");
    try {
      XSSFFormulaEvaluator xssfFormulaEvaluator =
          new XSSFFormulaEvaluator(this.sheet.getWorkbook());
      xssfFormulaEvaluator.evaluateAll();
    } catch (NotImplementedException e) {
      log.info(e);
    }
  }

  @Override
  public void computeCell(int column, int row) {
    // log.debug("Refreshing cell");
    try {
      XSSFCell xssfCell = customGetCell(column, row);
      XSSFFormulaEvaluator xssfFormulaEvaluator =
          new XSSFFormulaEvaluator(this.sheet.getWorkbook());
      xssfFormulaEvaluator.evaluateFormulaCell(xssfCell);
    } catch (NotImplementedException e) {
      log.info(e);
    }
  }

  @Override
  public double getValue(int column, int row) {
    computeCalculations();
    return getValueWithoutRecalculation(column, row);
  }

  @Override
  public double getValueWithoutRecalculation(int column, int row) {
    double returnValue = -99999;
    try {
      returnValue = customGetCell(column, row).getNumericCellValue();
    } catch (IllegalStateException e) {
      log.debug("Couldn't get a numerical value for cell {},{} in {}", column, row, this.name);
    }
    return returnValue;
  }

  public Object getCellContent(int column, int row) {
    XSSFCell cell = customGetCell(column, row);
    Object cellValue;

    try {
      if (cell.getRawValue() == null) {
        cellValue = "";
      } else {
        cellValue = cell.getNumericCellValue();
      }
    } catch (IllegalStateException | NotImplementedException e) {
      cellValue = cell.getRawValue();
    }

    return cellValue;
  }

  @Override
  public Object[][] getValues(int endColumn, int endRow) {

    try {
      XSSFFormulaEvaluator xssfFormulaEvaluator =
          new XSSFFormulaEvaluator(this.sheet.getWorkbook());
      xssfFormulaEvaluator.evaluateAll();
    } catch (NotImplementedException e) {
      log.info(e);
      log.info(e.getMessage());
      log.info(e.getCause());
    }

    Object[][] returnObject = new Object[endRow + 1][endColumn + 1];
    for (int i = 0; i <= endRow; i++) {
      for (int j = 0; j <= endColumn; j++) {
        returnObject[i][j] = getCellContent(j, i);
      }
    }

    return returnObject;
  }

  @Override
  public String getErrorInCell(int column, int row) {
    String error = null;
    XSSFCell cell = customGetCell(column, row);
    if (cell.getCellComment() != null) {
      error = cell.getCellComment().getString().toString();
    }
    return error;
  }

  @Override
  public void setRowValues(int row, String[] data) {
    for (int i = 0; i < data.length; i++) {
      setFormula(data[i], i, row);
    }
  }
}
