package at.asit.grapheme_clusters

internal val GRAPHEME_CLUSTERS = Regex("\\X")
internal const val ELLIPSIS = "…"

/**
 * The String as a sequence of **UTF-8 encoded bytes**.
 */
val String.bytes get() = this.encodeToByteArray()

/**
 * The length of the String, measured in **UTF-8 encoded bytes**.
 */
val String.byteLength get() = this.bytes.size

/**
 * The String as a sequence of **grapheme clusters**.
 *
 * (A **grapheme cluster** is a single user-perceived character.)
 */
val String.graphemeClusters get() = GRAPHEME_CLUSTERS.findAll(this).map { it.value }

/**
 * The length of the String, measured in **grapheme clusters**.
 *
 * (A **grapheme cluster** is a single user-perceived character.)
 */
val String.graphemeLength get() = this.graphemeClusters.count()

/**
 * Truncates the string along a grapheme cluster boundary.
 * The resulting String, including the cutoff indicator, will have a length measure less than or equal to the target length.
 *
 * (A **grapheme cluster** is a single user-perceived character.)
 *
 * @see String.truncateToByteLength
 * @see String.truncateToGraphemeLength
 */
fun String.truncate(targetLength: Int, lengthMeasure: (String) -> Int, cutoffIndicator: String = ELLIPSIS) : String {
    require(targetLength >= 0) { "targetLength must not be negative (was ${targetLength})" }

    // shortcut cases
    if (targetLength == 0) return ""
    if (lengthMeasure(this) <= targetLength) return this

    var currentLength = lengthMeasure(cutoffIndicator)
    if (currentLength > targetLength) return ""

    val builder = StringBuilder()
    for (cluster in this.graphemeClusters) {
        currentLength += lengthMeasure(cluster)
        if (currentLength > targetLength) break
        builder.append(cluster)
    }
    builder.append(cutoffIndicator)
    return builder.toString()
}

/**
 * Truncates the string along a grapheme cluster boundary.
 * The resulting String, including the cutoff indicator, will have a UTF-8 encoded byte length less than or equal to the target length.
 *
 * A potential terminating null byte is **not** accounted for. If this is desired, pass `targetLength-1` instead.
 *
 * (A **grapheme cluster** is a single user-perceived character.)
 *
 * @see String.truncate
 * @see String.truncateToGraphemeLength
 */
fun String.truncateToByteLength(targetLength: Int, cutoffIndicator: String = ELLIPSIS) =
    this.truncate(targetLength, String::byteLength, cutoffIndicator)

/**
 * Truncates the string along a grapheme cluster boundary.
 * The resulting String, including the cutoff indicator, will have a length measure less than or equal to the target length.
 *
 * (A **grapheme cluster** is a single user-perceived character.)
 *
 * @see String.truncate
 * @see String.truncateToByteLength
 */
fun String.truncateToGraphemeLength(targetLength: Int, cutoffIndicator: String = ELLIPSIS) =
    this.truncate(targetLength, String::graphemeLength, cutoffIndicator)
