package at.datenwort.firstClass.runtime;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

class StaticPropertyDefinition {
    private final static Map<Class<?>, Map<String, StaticPropertyDefinition>> staticPropertyDefinitions = new ConcurrentHashMap<>(1024);

    public final String fieldName;

    public final Class<?> modelClass;
    public final Field reflectField;
    public final Method getterMethod;
    public final Method setterMethod;

    private final Class<?> type;

    StaticPropertyDefinition(Class<?> modelClass, String fieldName) {
        String propName = Character.toUpperCase(fieldName.charAt(0)) + fieldName.substring(1);

        final String getterMethodName = "get" + propName;
        final String getterMethodNameIs = "is" + propName;
        final String setterMethodName = "set" + propName;

        Class<?> type = null;
        Field reflectField = null;
        try {
            reflectField = modelClass.getDeclaredField(fieldName);
            type = reflectField.getType();
        } catch (NoSuchFieldException e) {
            // ignore - seems to be just a getter-/setter property
        }

        Method getterMethod = null;
        try {
            getterMethod = modelClass.getDeclaredMethod(getterMethodName);
        } catch (NoSuchMethodException e) {
            try {
                getterMethod = modelClass.getDeclaredMethod(getterMethodNameIs);
            } catch (NoSuchMethodException e1) {
                // ignore
            }
        }

        if (type == null && getterMethod != null) {
            type = getterMethod.getReturnType();
        }

        Method setterMethod = null;
        try {
            setterMethod = findSetter(modelClass, setterMethodName, type);
            if (type == null) {
                type = setterMethod.getParameterTypes()[0];
            }
        } catch (NoSuchMethodException e) {
            // ignore - read-only property
        }

        if (type == null) {
            throw new IllegalStateException();
        }

        this.modelClass = modelClass;
        this.fieldName = fieldName;
        this.reflectField = reflectField;
        this.getterMethod = getterMethod;
        this.setterMethod = setterMethod;
        this.type = type;
    }

    private Method findSetter(Class<?> modelClass, String setterMethodName, Class<?> type) throws NoSuchMethodException {
        if (type != null) {
            return modelClass.getDeclaredMethod(setterMethodName, type);
        }

        /*
         * for the given property, we did not yet find a field or a getter, therefore, we did not know the type above
         * and have to go the hard route now.
         */
        for (Method method : modelClass.getDeclaredMethods()) {
            if (method.getParameterTypes().length != 1 || !setterMethodName.equals(method.getName())) {
                continue;
            }

            return method;
        }

        throw new NoSuchMethodException(setterMethodName);
    }

    public Class<?> getType() {
        return type;
    }

    public static StaticPropertyDefinition getDefinition(Class<?> declaringClass, String name) {
        Map<String, StaticPropertyDefinition> propertyDefinitions = staticPropertyDefinitions
                .computeIfAbsent(declaringClass, _ -> new ConcurrentHashMap<>(256));

        StaticPropertyDefinition propertyDefinition = propertyDefinitions
                .computeIfAbsent(name, n -> new StaticPropertyDefinition(declaringClass, n));

        return propertyDefinition;
    }
}
