package at.datenwort.firstClass.runtime.aspect;

import at.datenwort.firstClass.runtime.FcProperty;
import at.datenwort.firstClass.runtime.FcUtils;
import at.datenwort.firstClass.runtime.FirstClass;
import at.datenwort.firstClass.runtime.FirstClassConfiguration;
import at.datenwort.firstClass.runtime.PropertyChangeLazyInit;
import at.datenwort.firstClass.runtime.PropertyChangeListenerEvent;
import at.datenwort.firstClass.runtime.PropertyChangeListenerEventComparator;
import at.datenwort.firstClass.runtime.PropertyChangeListenerEvent_PCE;
import at.datenwort.firstClass.runtime.PropertySupportInterface;
import at.datenwort.firstClass.runtime.annotations.IgnoreParentHandling;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.WeakHashMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;

/**
 * Utility class, similar to Java's PropertyChangeSupport, just that it uses a more sophisticated
 * equality check. {@see #firePropertyChange}
 */
public class PropertyChangeSupport implements PropertySupportInterface {
    private final static String ALL_PROPERTIES = "";

    private final PropertySupportInterface source;

    private volatile Map<PropertyChangeSupport, String> _$_parents;

    private boolean _$_firstSetInited;
    private boolean _$_hasListener;

    private Map<String, Collection<PropertyChangeListener>> listeners;

    public PropertyChangeSupport(PropertySupportInterface sourceBean) {
        source = sourceBean;
    }

    @Override
    public boolean firstSet() {
        if (!_$_firstSetInited) {
            _$_firstSetInited = true;
            return true;
        }

        return false;
    }

    @Override
    public void lazySetup() {
        if (source instanceof PropertyChangeLazyInit propertyChangeLazyInit) {
            if (_$_parents != null) {
                _$_parents.keySet()
                        .forEach(parent -> {
                            if (parent.firstSet()) {
                                parent.lazySetup();
                            }
                        });
            }

            propertyChangeLazyInit.propertyChangeLazyInit();
        }
    }

    private void firePropertyChange(boolean ignoreParentHandling, PropertyChangeEvent event) {
        Object oldValue = event.getOldValue();
        Object newValue = event.getNewValue();
        if (!checkEquals(oldValue, newValue)) {
            if (!ignoreParentHandling && newValue instanceof PropertySupportInterface newValuePropertySupportInterface) {
                setupParent(null, this, event.getPropertyName(), newValuePropertySupportInterface);
            }

            Set<PropertyChangeListenerEvent> collectedEvents = _propagateChange(null, null, event);
            if (collectedEvents != null) {
                List<PropertyChangeListenerEvent> sortedEvents = new ArrayList<>(collectedEvents);
                sortedEvents.sort(PropertyChangeListenerEventComparator.INSTANCE);
                sortedEvents.forEach(PropertyChangeListenerEvent::fire);
            }

            if (oldValue instanceof PropertySupportInterface propertySupportInterface
                    && propertySupportInterface._propertyChangeSupport()._$_parents != null) {
                propertySupportInterface._propertyChangeSupport()._$_parents.remove(this);
            }
        }
    }

    private void setupParent(Set<Object> seenChildren, PropertyChangeSupport pcsParent, String propertyName, PropertySupportInterface newValue) {
        PropertyChangeSupport pcs = newValue._propertyChangeSupport();
        pcs._prepareParents();

        if (pcsParent != null
                && propertyName != null) {
            pcs._$_parents.put(pcsParent, propertyName);
        }

        FirstClass<?> fc = FcUtils.getFirstClass(newValue.getClass());
        if (fc == null) {
            return;
        }

        Collection<FcProperty<Object>> fcProperties = fc.getProperties();
        for (FcProperty<?> field : fcProperties) {
            if (!PropertySupportInterface.class.isAssignableFrom(field.getType())) {
                continue;
            }
            if (!field.isReadable(newValue, true)) {
                continue;
            }
            if (field.getAnnotation(IgnoreParentHandling.class) != null) {
                continue;
            }

            Object fieldValue = field.get(newValue, true);
            if (fieldValue == null) {
                continue;
            }

            if (seenChildren == null) {
                seenChildren = Collections.newSetFromMap(new IdentityHashMap<>());
            }

            if (seenChildren.contains(fieldValue)) {
                continue;
            }
            seenChildren.add(fieldValue);

            setupParent(seenChildren, pcs, field.getName(), (PropertySupportInterface) fieldValue);
        }
    }

    private void _prepareParents() {
        if (_$_parents == null) {
            synchronized (this) {
                if (_$_parents == null) {
                    _$_parents = Collections.synchronizedMap(new WeakHashMap<>());
                }
            }
        }
    }

    @Override
    public void firePropertyChange(String name, Object oldValue, Object newValue, boolean ignoreParentHandling) {
        firePropertyChange(ignoreParentHandling, new PropertyChangeEvent(this.source, name, oldValue, newValue));
    }

    private Set<PropertyChangeListenerEvent> _firePropertyChange(Set<PropertyChangeListenerEvent> events, PropertyChangeEvent event) {
        if (listeners != null && !listeners.isEmpty()) {
            events = _firePropertyChange(events, event, listeners.entrySet());
        }

        return events;
    }

    private Set<PropertyChangeListenerEvent> _firePropertyChange(Set<PropertyChangeListenerEvent> events,
                                                                 PropertyChangeEvent event,
                                                                 Set<Map.Entry<String, Collection<PropertyChangeListener>>> listenerEntries) {
        final String propertyNameRoot = event.getPropertyName() + ".";

        nextPropertyListener:
        for (Map.Entry<String, Collection<PropertyChangeListener>> listenerEntry : listenerEntries) {
            final String listenerPropertyName = listenerEntry.getKey();
            final Collection<PropertyChangeListener> listeners = listenerEntry.getValue();

            for (PropertyChangeListener listener : listeners) {
                if (Objects.equals(listenerPropertyName, ALL_PROPERTIES)
                        || listenerPropertyName.equals(event.getPropertyName())) {
                    // actual notification 1
                    events = addEvent(events, new PropertyChangeListenerEvent_PCE(listener, event));
                } else if (listenerPropertyName.startsWith(propertyNameRoot)) {
                    if (event.getOldValue() != null || event.getNewValue() != null) {
                        String propertyName = listenerPropertyName.substring(event.getPropertyName()
                                .length() + 1);

                        Object oldBaseValue = event.getOldValue();
                        Object newBaseValue = event.getNewValue();

                        Object oldPropValue = null;
                        Object newPropValue = null;

                        if (oldBaseValue != null) {
                            FirstClass<?> fc = FcUtils.getFirstClass(oldBaseValue.getClass());
                            FcProperty<?> property = FcUtils.findProperty(fc, propertyName);
                            if (property != null) {
                                oldPropValue = property.get(oldBaseValue);
                            }
                        }
                        if (newBaseValue != null) {
                            FirstClass<?> fc = FcUtils.getFirstClass(newBaseValue.getClass());
                            FcProperty<?> property = FcUtils.findProperty(fc, propertyName);
                            if (property != null) {
                                newPropValue = property.get(newBaseValue);
                            }
                        }

                        if (!checkEquals(oldPropValue, newPropValue)) {
                            events = _firePropertyChange(events,
                                    new PropertyChangeEvent(
                                            this.source,
                                            listenerPropertyName,
                                            oldPropValue,
                                            newPropValue));
                        }
                    }

                    continue nextPropertyListener;
                }
            }
        }

        return events;
    }

    private Set<PropertyChangeListenerEvent> addEvent(Set<PropertyChangeListenerEvent> events, PropertyChangeListenerEvent propertyChangeListenerEvent) {
        if (events == null) {
            events = new HashSet<>();
        }

        events.add(propertyChangeListenerEvent);

        return events;
    }

    private Set<PropertyChangeListenerEvent> _propagateChange(Set<PropertyChangeListenerEvent> events,
                                                              Set<PropertyChangeSupport> seenObjects,
                                                              PropertyChangeEvent event) {
        events = _firePropertyChange(events, event);

        if (_$_parents != null) {
            for (Map.Entry<PropertyChangeSupport, String> parents : new ArrayList<>(_$_parents.entrySet())) {
                if (parents.getKey() == this) {
                    // avoid endless loop if datastructure contains circular references
                    continue;
                }

                if (seenObjects == null) {
                    seenObjects = Collections.newSetFromMap(new IdentityHashMap<>());
                }

                if (seenObjects.contains(parents.getKey())) {
                    // avoid endless loop if datastructure contains circular references
                    continue;
                }
                seenObjects.add(parents.getKey());

                String propertyName = parents.getValue() + "." + event.getPropertyName();

                events = parents.getKey()
                        ._propagateChange(
                                events,
                                seenObjects,
                                new PropertyChangeEvent(
                                        event.getSource(),
                                        propertyName,
                                        event.getOldValue(),
                                        event.getNewValue()
                                ));
            }
        }

        return events;
    }

    @SuppressWarnings("BooleanMethodIsAlwaysInverted")
    private boolean checkEquals(Object oldValue, Object newValue) {
        Boolean ret = FirstClassConfiguration.equalityHandler.equals(oldValue, newValue);
        if (ret == null) {
            throw new IllegalStateException("equality check failed. no result. Old value=" + oldValue + " New value=" + newValue);
        }

        return ret;
    }

    @Override
    public boolean hasListeners() {
        return _$_hasListener || _$_parents != null;
    }

    @Override
    public void addPropertyChangeListener(PropertyChangeListener pcl) {
        _$_hasListener = true;
        if (listeners == null) {
            listeners = new ConcurrentHashMap<>();
        }

        Collection<PropertyChangeListener> listenerQueue = listeners.computeIfAbsent(ALL_PROPERTIES, _ -> new ConcurrentLinkedQueue<>());
        listenerQueue.add(pcl);
    }

    @Override
    public void addPropertyChangeListener(String propertyName, PropertyChangeListener pcl) {
        _$_hasListener = true;
        if (listeners == null) {
            listeners = new ConcurrentHashMap<>();
        }

        Collection<PropertyChangeListener> listenerQueue = listeners.computeIfAbsent(propertyName, _ -> new ConcurrentLinkedQueue<>());
        listenerQueue.add(pcl);
    }

    @Override
    public void removePropertyChangeListener(PropertyChangeListener pcl) {
        Collection<PropertyChangeListener> listenerQueue = listeners == null ? null : listeners.get(ALL_PROPERTIES);
        if (listenerQueue != null) {
            listenerQueue.remove(pcl);
        }
    }

    @Override
    public void removePropertyChangeListener(String propertyName, PropertyChangeListener pcl) {
        Collection<PropertyChangeListener> listenerQueue = listeners == null ? null : listeners.get(propertyName);
        if (listenerQueue != null) {
            listenerQueue.remove(pcl);
        }
    }

    @Override
    public PropertyChangeSupport _propertyChangeSupport() {
        return this;
    }
}
