package at.datenwort.firstClass.runtime.aspect;

import at.datenwort.commons.agentsFromService.api.AgentConfigurator;
import at.datenwort.firstClass.runtime.PropertySupportInterface;
import at.datenwort.firstClass.runtime.annotations.IgnorePropertyChange;
import jakarta.annotation.Nonnull;
import net.bytebuddy.agent.builder.AgentBuilder;
import net.bytebuddy.asm.AsmVisitorWrapper;
import net.bytebuddy.description.NamedElement;
import net.bytebuddy.description.field.FieldDescription;
import net.bytebuddy.description.field.FieldList;
import net.bytebuddy.description.modifier.FieldManifestation;
import net.bytebuddy.description.modifier.FieldPersistence;
import net.bytebuddy.description.modifier.Ownership;
import net.bytebuddy.description.modifier.Visibility;
import net.bytebuddy.description.type.TypeDescription;
import net.bytebuddy.dynamic.DynamicType;
import net.bytebuddy.implementation.Implementation;
import net.bytebuddy.implementation.MethodDelegation;
import net.bytebuddy.implementation.bytecode.constant.DefaultValue;
import net.bytebuddy.jar.asm.ClassWriter;
import net.bytebuddy.jar.asm.Opcodes;
import net.bytebuddy.matcher.ElementMatchers;
import net.bytebuddy.utility.OpenedClassReader;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.stream.Collectors;

public class PropertyChangeSupportWeaver implements AgentConfigurator {
    @Override
    public int getOrder() {
        return 300;
    }

    public AgentBuilder install(AgentBuilder agentBuilder) {
        final ClassExtractor classExtractor = new ClassExtractor();
        final List<ClassExtractor.MethodApplier> propertyChangeSupportMixinMethods = classExtractor.extractMethods(PropertyChangeSupportMixin.class);

        final Set<String> propertyChangeSupportWoven = ConcurrentHashMap.newKeySet();

        return agentBuilder.type(ElementMatchers.isAnnotatedWith(at.datenwort.firstClass.runtime.annotations.PropertyChangeSupport.class)
                )
                .transform(((builder, typeDescription, classLoader, module, protectionDomain) -> {
                    try {
                        String className = typeDescription.getName();
                        DynamicType.Builder<?> ret = builder;

                        if (!isClassOrSuperClassAlreadyWoven(propertyChangeSupportWoven, typeDescription)) {
                            ret = weavePropertyChangeSupport(typeDescription, ret, propertyChangeSupportMixinMethods);
                            propertyChangeSupportWoven.add(className);
                        }

                        ret = weaveFieldSetters(typeDescription, ret);
                        ret = weaveFieldSetIntercept(typeDescription, ret);

                        ret = ret.visit(
                                new AsmVisitorWrapper.ForDeclaredMethods()
                                        .writerFlags(ClassWriter.COMPUTE_MAXS));
                        return ret;
                    } catch (Exception e) {
                        throw new RuntimeException(e);
                    }
                }));
    }

    private boolean isClassOrSuperClassAlreadyWoven(Set<String> propertyChangeSupportWoven, TypeDescription typeDescription) {
        TypeDescription.Generic checkType = typeDescription.asGenericType();
        do {
            if (propertyChangeSupportWoven.contains(checkType.getTypeName())) {
                return true;
            }
            checkType = checkType.getSuperClass();
        }
        while (checkType != null);

        return false;
    }

    private @Nonnull DynamicType.Builder<?> weaveFieldSetters(TypeDescription typeDescription, DynamicType.Builder<?> builder) {
        // install the "write field" interceptors
        FieldList<FieldDescription.InDefinedShape> fieldList = typeDescription.getDeclaredFields()
                .filter(ElementMatchers.not(
                        ElementMatchers.isAnnotatedWith(IgnorePropertyChange.class)
                                .or(ElementMatchers.isStatic())));
        for (FieldDescription.InDefinedShape field : fieldList) {
            builder = builder.defineMethod("_$_set_" + field.getName(),
                            field.getType()
                                    .asErasure(),
                            Visibility.PROTECTED,
                            Ownership.STATIC)
                    .withParameter(field.getType()
                                    .asErasure(),
                            "newValue")
                    .withParameter(field.getType()
                                    .asErasure(),
                            "oldValue")
                    .withParameter(TypeDescription.ForLoadedType.of(PropertySupportInterface.class),
                            "target")
                    .withParameter(TypeDescription.ForLoadedType.of(String.class),
                            "fieldName")
                    .withParameter(TypeDescription.ForLoadedType.of(boolean.class),
                            "registerOnly")
                    .intercept(MethodDelegation.to(PropertyChangeSupportSetFieldInterceptor.class));
        }
        return builder;
    }

    private @Nonnull DynamicType.Builder<?> weaveFieldSetIntercept(TypeDescription typeDescription, DynamicType.Builder<?> builder) {
        // install the "write field" interceptors
        FieldList<FieldDescription.InDefinedShape> fieldList = typeDescription.getDeclaredFields()
                .filter(ElementMatchers.not(
                        ElementMatchers.isAnnotatedWith(IgnorePropertyChange.class)
                                .or(ElementMatchers.isStatic())));
        Map<String, FieldDescription.InDefinedShape> fieldMaps = fieldList.stream()
                .collect(Collectors.toMap(NamedElement.WithRuntimeName::getName, Function.identity()));
        builder = builder.visit(
                new AsmVisitorWrapper.ForDeclaredMethods()
                        .constructor(ElementMatchers.any(), (instrumentedType, instrumentedMethod, methodVisitor, implementationContext, typePool, writerFlags, readerFlags) ->
                                new InspectingMethodVisitor(OpenedClassReader.ASM_API, methodVisitor) {

                                    @Override
                                    public void visitFieldInsn(int opcode, String owner, String name, String descriptor) {
                                        if (opcode != Opcodes.PUTFIELD) {
                                            super.visitFieldInsn(opcode, owner, name, descriptor);
                                            return;
                                        }

                                        FieldDescription.InDefinedShape fieldShape = fieldMaps.get(name);
                                        if (fieldShape == null || !fieldShape.isFinal()) {
                                            super.visitFieldInsn(opcode, owner, name, descriptor);
                                            return;
                                        }

                                        insertRegisterMethod(this, fieldShape, implementationContext, opcode, owner, name, descriptor);
                                    }
                                })
                        .method(ElementMatchers.any()
                                        .and(ElementMatchers.not(ElementMatchers.nameContains("_$_")))
                                , (instrumentedType, instrumentedMethod, methodVisitor, implementationContext, typePool, writerFlags, readerFlags) ->
                                        new InspectingMethodVisitor(OpenedClassReader.ASM_API, methodVisitor) {
                                            @Override
                                            public void visitFieldInsn(int opcode, String owner, String name, String descriptor) {
                                                if (opcode != Opcodes.PUTFIELD) {
                                                    super.visitFieldInsn(opcode, owner, name, descriptor);
                                                    return;
                                                }

                                                FieldDescription.InDefinedShape fieldShape = fieldMaps.get(name);
                                                if (fieldShape == null) {
                                                    super.visitFieldInsn(opcode, owner, name, descriptor);
                                                    return;
                                                }

                                                insertInterceptMethod(this, opcode, owner, name, descriptor);
                                            }
                                        }
                        )
        );
        return builder;
    }

    private @Nonnull DynamicType.Builder<?> weavePropertyChangeSupport(TypeDescription typeDescription,
                                                                       DynamicType.Builder<?> ret,
                                                                       List<ClassExtractor.MethodApplier> propertyChangeSupportMixinMethods) {
        ret = ret.implement(PropertySupportInterface.class);

        /*
         create the local variable which will hold a reference to the complex _PropertyChangeSupport implementation.
         this local variable will be lazily initialized via PropertyChangeSupportMixin#_propertyChangeSupport()
         */
        ret = ret.defineField("_$_pcs", PropertyChangeSupport.class,
                Visibility.PRIVATE,
                FieldPersistence.TRANSIENT,
                FieldManifestation.VOLATILE);

        // mixin the "PropertyChangeSupportMixin" class
        for (ClassExtractor.MethodApplier ma : propertyChangeSupportMixinMethods) {
            ClassExtractor.MethodDefinition md = ma.getMethodDefinition();
            ret = ret.defineMethod(md.name(),
                            md.methodSignature()
                                    .returnType(),
                            md.access())
                    .withParameters(md.methodSignature()
                            .parameters())
                    .intercept(ma.forOwner(typeDescription.getInternalName())
                            .asImplementation());
        }

        return ret;
    }

    private void insertInterceptMethod(InspectingMethodVisitor methodVisitor,
                                       int opcode, String owner, String name, String descriptor) {
        // load current value
        methodVisitor.visitVarInsn(Opcodes.ALOAD, 0);
        methodVisitor._visitFieldInsn(Opcodes.GETFIELD, owner, name, descriptor);

        // load this
        methodVisitor.visitVarInsn(Opcodes.ALOAD, 0);

        // load property name
        methodVisitor.visitLdcInsn(name);

        // load "register only" flag
        methodVisitor.visitLdcInsn(false);

        // invoke setter interceptor
        methodVisitor.visitMethodInsn(Opcodes.INVOKESTATIC,
                owner,
                "_$_set_" + name,
                "("
                        + descriptor
                        + descriptor
                        + "L" + TypeDescription.ForLoadedType.of(PropertySupportInterface.class)
                        .getInternalName() + ";"
                        + "Ljava/lang/String;"
                        + "Z"
                        + ")"
                        + descriptor,
                false);

        // actually set the field
        methodVisitor._visitFieldInsn(opcode, owner, name, descriptor);
    }

    private void insertRegisterMethod(final InspectingMethodVisitor methodVisitor,
                                      final FieldDescription.InDefinedShape fieldShape,
                                      final Implementation.Context implementationContext,
                                      final int opcode,
                                      final String owner,
                                      final String name,
                                      final String descriptor) {
        // load a default value for the old value
        DefaultValue.of(fieldShape.getType())
                .apply(methodVisitor, implementationContext);

        // load this
        methodVisitor.visitVarInsn(Opcodes.ALOAD, 0);

        // load property name
        methodVisitor.visitLdcInsn(name);

        // load "register only" flag
        methodVisitor.visitLdcInsn(true);

        // invoke setter interceptor
        methodVisitor.visitMethodInsn(Opcodes.INVOKESTATIC,
                owner,
                "_$_set_" + name,
                "("
                        + descriptor
                        + descriptor
                        + "L" + TypeDescription.ForLoadedType.of(PropertySupportInterface.class)
                        .getInternalName() + ";"
                        + "Ljava/lang/String;"
                        + "Z"
                        + ")"
                        + descriptor,
                false);

        // actually set the field
        methodVisitor._visitFieldInsn(opcode, owner, name, descriptor);
    }
}
