package at.datenwort.firstClass.runtime;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static java.lang.String.join;

class FcFirstClassDefinition<T> extends FcPropertyDefinition<T> implements FirstClass<T> {
    private final Class<T> firstClassType;
    private final FirstClass<?> firstClassStruct;

    private Class<T> classType;
    private Map<String, FcProperty<Object>> propertiesByName;

    FcFirstClassDefinition(FcProperty<?> parentProperty,
                           Class<?> declaringClass, String name, Class<T> firstClassType,
                           FirstClass<?> firstClass) {
        super(parentProperty,
                declaringClass,
                name);
        this.firstClassType = firstClassType;
        if (!firstClassType.getName().endsWith("$fc")) {
            throw new IllegalArgumentException("First class must be an $fc class");
        }

        this.firstClassStruct = firstClass;
    }

    @Override
    public Class<T> getType() {
        if (classType == null) {
            //noinspection unchecked
            classType = (Class<T>) FcUtils.getFcOwnerClass(firstClassType);
        }
        return classType;
    }

    @Override
    public Class<T> getFirstClassType() {
        return firstClassType;
    }

    @Override
    public <DUMMY> FcProperty<DUMMY> findProperty(String name) {
        if (propertiesByName == null) {
            init();
        }

        //noinspection unchecked
        return (FcProperty<DUMMY>) propertiesByName.get(name);
    }

    @Override
    public Collection<FcProperty<Object>> getProperties() {
        if (propertiesByName == null) {
            init();
        }

        return propertiesByName.values();
    }

    private void init() {
        Map<String, FcProperty<Object>> propertiesByName;
        if (firstClassStruct == null) {
            propertiesByName = firstInit(firstClassType);
        } else {
            propertiesByName = nestedInit(firstClassStruct);
        }
        this.propertiesByName = Collections.unmodifiableMap(propertiesByName);
    }

    private Map<String, FcProperty<Object>> nestedInit(final FirstClass<?> firstClassStruct) {
        Map<String, FcProperty<Object>> propertiesByName = new LinkedHashMap<>();
        for (FcProperty<Object> property : firstClassStruct.getProperties()) {
            FcProperty<Object> nestedProperty = FirstClassFactory.createPropertyDefinition(
                    this,
                    property.getDeclaringClass(),
                    property.getName(),
                    property.getType()
            );
            propertiesByName.put(nestedProperty.getPropertyName(), nestedProperty);
        }
        return propertiesByName;
    }

    private Map<String, FcProperty<Object>> firstInit(final Class<?> firstClassType) {
        List<Class<?>> hierarchy = new ArrayList<>();

        {
            Class<?> propertyClass = firstClassType;
            do {
                if (hierarchy.isEmpty()) {
                    hierarchy.add(propertyClass);
                } else {
                    hierarchy.addFirst(propertyClass);
                }

                if (propertyClass.getInterfaces().length > 0) {
                    propertyClass = propertyClass.getInterfaces()[0];
                } else {
                    propertyClass = null;
                }
            }
            while (propertyClass != null &&
                    FirstClass.class.isAssignableFrom(propertyClass)
                    && FirstClass.class != propertyClass);
        }

        {
            Map<String, FcProperty<Object>> propertiesByName = new LinkedHashMap<>();

            for (Class<?> propertyClass : hierarchy) {
                Map<String, Method> declaredMethods = Stream.of(propertyClass.getDeclaredMethods()).collect(
                        Collectors.toMap(
                                Method::getName,
                                Function.identity(),
                                (u, _) -> {
                                    throw new IllegalStateException(String.format("Duplicate key %s", u));
                                },
                                LinkedHashMap::new));
                List<String> declaredPropertyNames;
                try {
                    //noinspection unchecked
                    declaredPropertyNames = (List<String>) propertyClass.getDeclaredField("declaredPropertyNames").get(0);
                } catch (IllegalAccessException | NoSuchFieldException e) {
                    throw new RuntimeException(e);
                }
                for (String propertyName : declaredPropertyNames) {
                    Method fcMethod = declaredMethods.get(propertyName);

                    if (fcMethod == null
                            || fcMethod.getParameterCount() != 0
                            || !FcProperty.class.isAssignableFrom(fcMethod.getReturnType())
                            || FcUtils.getFcOwnerClass(fcMethod.getDeclaringClass()) == null) {
                        continue;
                    }

                    //noinspection unchecked
                    FcProperty<Object> fcProperty = FirstClassFactory.createPropertyDefinition(
                            this,
                            FcUtils.getFcOwnerClass(fcMethod.getDeclaringClass()),
                            fcMethod.getName(),
                            (Class<Object>) fcMethod.getReturnType()
                    );
                    propertiesByName.put(fcProperty.getPropertyName(), fcProperty);
                }
            }

            return propertiesByName;
        }
    }

    @Override
    public String toString() {
        return join(".", firstClassType.getName(), getPropertyName());
    }
}
