package at.datenwort.firstClass.runtime;

import java.lang.annotation.Annotation;
import java.util.ArrayList;
import java.util.List;

class FcPropertyDefinition<T> implements FcProperty<T> {
    private List<FcProperty<?>> propertyPath;
    private final FcProperty<?> parentProperty;
    private final StaticPropertyDefinition definition;
    private transient String propName;

    FcPropertyDefinition(FcProperty<?> parentProperty, Class<?> declaringClass, String name) {
        this.parentProperty = parentProperty;
        if (declaringClass != null) {
            this.definition = StaticPropertyDefinition.getDefinition(declaringClass, name);
        } else {
            this.definition = null;
        }
    }

    @Override
    public FcProperty<?> getParent() {
        return parentProperty;
    }

    @Override
    public Class<?> getDeclaringClass() {
        return definition != null ? definition.getDeclaringClass() : null;
    }

    public Class<T> getType() {
        //noinspection unchecked
        return definition != null ? (Class<T>) definition.getType() : null;
    }

    public T get(Object obj, boolean direct) {
        obj = resolve(obj);
        return _get(obj, direct);
    }

    private T _get(Object obj, boolean direct) {
        if (obj == null || definition == null) {
            //noinspection unchecked
            return (T) obj;
        }

        try {
            if (!direct && definition.getterMethod != null) {
                return FirstClassConfiguration.propertyGetter.getFieldMethod(this, definition.getterMethod, obj);
            }

            return FirstClassConfiguration.propertyGetter.getField(this, definition.reflectField, obj);
        } catch (Throwable e) {
            throw new RuntimeException(e);
        }
    }

    private Object resolve(Object obj) {
        if (obj == null) {
            return null;
        }

        if (propertyPath == null) {
            List<FcProperty<?>> properties = new ArrayList<>();

            FcProperty<?> parentProperty = this.parentProperty;
            while (parentProperty != null) {
                properties.addFirst(parentProperty);
                parentProperty = parentProperty.getParent();
            }

            this.propertyPath = properties;
        }

        if (propertyPath.isEmpty()) {
            return obj;
        }

        for (FcProperty<?> fcProperty : propertyPath) {
            obj = ((FcPropertyDefinition<?>) fcProperty)._get(obj, false);
            if (obj == null) {
                return null;
            }
        }

        return obj;
    }

    public void set(Object obj, T value, boolean direct) {
        obj = resolve(obj);
        _set(obj, value, direct);
    }

    @Override
    public boolean isWriteable(Object obj, boolean direct) {
        if (definition == null) {
            return false;
        }

        if (!direct && definition.setterMethod == null) {
            return false;
        }

        if (definition.reflectField == null) {
            return false;
        }

        obj = resolve(obj);
        return obj != null;
    }

    @Override
    public boolean isReadable(Object obj, boolean direct) {
        if (definition == null) {
            return false;
        }

        if (!direct && definition.getterMethod == null) {
            return false;
        }

        if (definition.reflectField == null) {
            return false;
        }

        obj = resolve(obj);
        return obj != null;
    }

    @Override
    public boolean isWriteable() {
        return definition != null
                && (definition.reflectField != null || definition.setterMethod != null);
    }

    @Override
    public boolean isReadable() {
        return definition != null
                && (definition.reflectField != null || definition.getterMethod != null);
    }

    private void _set(Object obj, T value, boolean direct) {
        if (obj == null) {
            return;
        }

        try {
            if (!direct && definition.setterMethod != null) {
                FirstClassConfiguration.propertySetter.setFieldMethod(this, definition.setterMethod, obj, value);
                return;
            }

            FirstClassConfiguration.propertySetter.setField(this, definition.reflectField, obj, value);
        } catch (Throwable e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public String getName() {
        if (definition == null) {
            return "";
        }

        if (propName == null) {
            StringBuilder sb = new StringBuilder();
            if (parentProperty != null) {
                sb.append(parentProperty.getName());
            }

            if (!sb.isEmpty()) {
                sb.append(".");
            }

            sb.append(definition.fieldName);

            propName = sb.toString();
        }

        return propName;
    }

    @Override
    public String getPropertyName() {
        return definition != null ? definition.fieldName : null;
    }

    @Override
    public <A extends Annotation> A getAnnotation(Class<A> annotationClass) {
        if (definition.reflectField != null) {
            return definition.reflectField.getAnnotation(annotationClass);
        }

        if (definition.getterMethod != null) {
            return definition.getterMethod.getAnnotation(annotationClass);
        }

        if (definition.setterMethod != null) {
            return definition.setterMethod.getAnnotation(annotationClass);
        }

        return null;
    }

    @Override
    public boolean isBackedByField() {
        return definition.reflectField != null;
    }

    public String toString() {
        return definition != null ? (definition.modelClass.getName() + " " + definition.fieldName) : "?";
    }
}
