package at.datenwort.firstClass.runtime.aspect;

import jakarta.annotation.Nonnull;
import net.bytebuddy.description.method.MethodDescription;
import net.bytebuddy.description.type.TypeDescription;
import net.bytebuddy.implementation.Implementation;
import net.bytebuddy.implementation.bytecode.ByteCodeAppender;
import net.bytebuddy.jar.asm.ClassReader;
import net.bytebuddy.jar.asm.ClassVisitor;
import net.bytebuddy.jar.asm.Label;
import net.bytebuddy.jar.asm.MethodVisitor;
import net.bytebuddy.jar.asm.signature.SignatureReader;
import net.bytebuddy.jar.asm.signature.SignatureVisitor;
import net.bytebuddy.pool.TypePool;
import net.bytebuddy.utility.OpenedClassReader;

import java.io.IOException;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.BiConsumer;

class ClassExtractor {

    record MethodDefinition(int access, String name, String descriptor, String signature, String[] exceptions,
                            MethodSignature methodSignature) {
    }

    static class MethodApplier {
        private final MethodDefinition methodDefinition;
        private final String owner;
        private final String newOwner;
        private final List<BiConsumer<MethodApplier, MethodVisitor>> operations;

        private MethodApplier(String owner, MethodDefinition methodDefinition) {
            this(owner, null, methodDefinition, new ArrayList<>());
        }

        private MethodApplier(String owner, String newOwner,
                              MethodDefinition methodDefinition,
                              List<BiConsumer<MethodApplier, MethodVisitor>> operations) {
            this.owner = owner;
            this.newOwner = newOwner;
            this.methodDefinition = methodDefinition;
            this.operations = operations;
        }

        MethodDefinition getMethodDefinition() {
            return methodDefinition;
        }

        void apply(MethodVisitor methodVisitor) {
            operations.forEach(operation -> operation.accept(this, methodVisitor));
        }

        MethodApplier forOwner(String name) {
            return new MethodApplier(owner, name, methodDefinition, operations);
        }

        String rewriteOwner(String owner) {
            if (newOwner == null || !this.owner.equals(owner)) {
                return owner;
            }

            return newOwner;
        }

        Implementation asImplementation() {
            return new Implementation.Simple((methodVisitor, _, _) -> {
                apply(methodVisitor);
                return ByteCodeAppender.Size.ZERO;
            });
        }
    }

    @SuppressWarnings("SameParameterValue")
    List<MethodApplier> extractMethods(Class<?> clazz) {
        final List<MethodApplier> allMethodOperations = new ArrayList<>();

        ClassReader cr;
        try {
            cr = new ClassReader(clazz.getName());
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        cr.accept(new ClassVisitor(OpenedClassReader.ASM_API) {
            @Override
            public MethodVisitor visitMethod(int access, String name, String descriptor, String signature, String[] exceptions) {
                if (MethodDescription.CONSTRUCTOR_INTERNAL_NAME.equals(name)
                        || !Modifier.isPublic(access)
                        || Modifier.isStatic(access)) {
                    return super.visitMethod(access, name, descriptor, signature, exceptions);
                }

                new SignatureReader(Objects.requireNonNullElse(signature, descriptor)).accept(new SignatureVisitor(OpenedClassReader.ASM_API) {
                    @Override
                    public SignatureVisitor visitParameterType() {
                        return super.visitParameterType();
                    }

                    @Override
                    public SignatureVisitor visitReturnType() {
                        return super.visitReturnType();
                    }
                });

                final MethodApplier methodApplier = new MethodApplier(cr.getClassName(), toMethodDefinition(access, name, descriptor, signature, exceptions));

                List<BiConsumer<MethodApplier, MethodVisitor>> methodOperations = methodApplier.operations;

                allMethodOperations.add(methodApplier);
                return new MethodVisitor(OpenedClassReader.ASM_API, super.visitMethod(access, name, descriptor, signature, exceptions)) {
                    @Override
                    public void visitCode() {
                        methodOperations.add((_, mv) -> mv.visitCode());
                    }

                    @Override
                    public void visitMaxs(int maxStack, int maxLocals) {
                        methodOperations.add((_, mv) -> mv.visitMaxs(maxStack, maxLocals));
                    }

                    @Override
                    public void visitEnd() {
                        methodOperations.add((_, mv) -> mv.visitEnd());
                    }

                    @Override
                    public void visitLabel(Label label) {
                        methodOperations.add((_, mv) -> mv.visitLabel(label));
                    }

                    @Override
                    public void visitTryCatchBlock(Label start, Label end, Label handler, String type) {
                        methodOperations.add((_, mv) -> mv.visitTryCatchBlock(start, end, handler, type));
                    }

                    @Override
                    public void visitInsn(int opcode) {
                        methodOperations.add((_, mv) -> mv.visitInsn(opcode));
                    }

                    @Override
                    public void visitVarInsn(int opcode, int varIndex) {
                        methodOperations.add((_, mv) -> mv.visitVarInsn(opcode, varIndex));
                    }

                    @Override
                    public void visitFieldInsn(int opcode, String owner, String name, String descriptor) {
                        methodOperations.add((ma, mv) -> mv.visitFieldInsn(opcode, ma.rewriteOwner(owner), name, descriptor));
                    }

                    @Override
                    public void visitJumpInsn(int opcode, Label label) {
                        methodOperations.add((_, mv) -> mv.visitJumpInsn(opcode, label));
                    }

                    @Override
                    public void visitFrame(int type, int numLocal, Object[] local, int numStack, Object[] stack) {
                        methodOperations.add((_, mv) -> mv.visitFrame(type, numLocal, local, numStack, stack));
                    }

                    @Override
                    public void visitLocalVariable(String name, String descriptor, String signature, Label start, Label end, int index) {
                        methodOperations.add((_, mv) -> mv.visitLocalVariable(name, descriptor, signature, start, end, index));
                    }

                    @Override
                    public void visitTypeInsn(int opcode, String type) {
                        methodOperations.add((_, mv) -> mv.visitTypeInsn(opcode, type));
                    }

                    @Override
                    public void visitMethodInsn(int opcode, String owner, String name, String descriptor, boolean isInterface) {
                        methodOperations.add((ma, mv) -> mv.visitMethodInsn(opcode, ma.rewriteOwner(owner), name, descriptor, isInterface));
                    }

                };
            }
        }, ClassReader.SKIP_DEBUG);

        return allMethodOperations;
    }

    MethodDefinition toMethodDefinition(int access, String name, String descriptor, String signature, String[] exceptions) {
        return new MethodDefinition(access, name, descriptor, signature, exceptions, parseJVMMethodSignature(descriptor));
    }

    record MethodSignature(TypeDescription returnType, List<TypeDescription> parameters) {
    }

    private final Map<String, MethodSignature> methodSignatures = new ConcurrentHashMap<>();

    private static class MutableInt {
        private int value;

        private int intValue() {
            return this.value;
        }

        private void increment() {
            this.value++;
        }
    }

    private MethodSignature parseJVMMethodSignature(String signature) {
        MethodSignature ret = methodSignatures.computeIfAbsent(signature, _ -> {

            List<TypeDescription> parameters = new ArrayList<>();
            TypeDescription returnType;

            MutableInt currentIndex = new MutableInt();
            char[] chars = signature.toCharArray();

            // Parsing the parameter part
            if (chars[currentIndex.intValue()] == '(') {
                currentIndex.increment();
                while (chars[currentIndex.intValue()] != ')') {
                    parameters.add(parseType(chars, currentIndex));
                    while (chars[currentIndex.intValue()] == '[' || chars[currentIndex.intValue()] == ';') {
                        currentIndex.increment();
                    }
                }
                currentIndex.increment(); // Skip the closing bracket
            }

            // Parsing the return type
            returnType = parseType(chars, currentIndex);

            return new MethodSignature(
                    returnType,
                    parameters);
        });

        return ret;
    }

    private static TypeDescription parseType(char[] chars, MutableInt currentIndex) {
        char firstChar = chars[currentIndex.intValue()];
        if (firstChar == '[') {
            // Array type
            currentIndex.increment();
            return toTypeDescription("[" + parseType(chars, currentIndex) + "]");
        } else if (firstChar == 'L') {
            // Object type (e.g., Ljava/lang/String;)
            StringBuilder type = new StringBuilder();
            currentIndex.increment(); // Skip the 'L'
            while (chars[currentIndex.intValue()] != ';') {
                type.append(chars[currentIndex.intValue()] == '/' ? '.' : chars[currentIndex.intValue()]);
                currentIndex.increment();
            }
            return toTypeDescription(type.toString());
        } else {
            // Primitive type
            TypeDescription type = toTypeDescription(mapType(firstChar));
            currentIndex.increment();
            return type;
        }
    }

    @Nonnull
    private static TypeDescription toTypeDescription(String type) {
        TypeDescription ret = TypePool.Default.ofSystemLoader()
                .describe(type)
                .resolve();
        return ret;
    }

    private final static Map<String, String> descriptorToType;

    static {
        descriptorToType = Map.of(
                Void.TYPE.descriptorString(), Void.TYPE.getName(),
                Boolean.TYPE.descriptorString(), Boolean.TYPE.getName(),
                Byte.TYPE.descriptorString(), Byte.TYPE.getName(),
                Short.TYPE.descriptorString(), Short.TYPE.getName(),
                Character.TYPE.descriptorString(), Character.TYPE.getName(),
                Integer.TYPE.descriptorString(), Integer.TYPE.getName(),
                Long.TYPE.descriptorString(), Long.TYPE.getName(),
                Double.TYPE.descriptorString(), Double.TYPE.getName(),
                Float.TYPE.descriptorString(), Float.TYPE.getName()
        );
    }

    private static String mapType(char descriptor) {
        String ret = descriptorToType.get(Character.toString(descriptor));
        if (ret == null) {
            throw new IllegalArgumentException("Unknown descriptor: " + descriptor);
        }
        return ret;
    }
}
