#!/usr/bin/bash
usage(){
cat << EOF
usage: 
	$0 clean
	
	Deletes the output directory.
	
	$0 <widoco-jar> [only-ont]

	Generates (or re-generates) all the ontology documentation.
	Parameters: 
		widoco-jar: the path to the widoco jar-with-dependencies to be run
		only-ont: generate docs only for the specified ontology
	Hint: Widoco can be downloaded from https://github.com/dgarijo/Widoco
EOF
}
script_path="$( cd "$( dirname "${BASH_SOURCE[0]}" )" >/dev/null 2>&1 && pwd )"
project_dir=${script_path}/../../../../../
project_dir="$( cd ${project_dir} >/dev/null 2>&1 && pwd )"
output_base=${project_dir}/target
if [[ $1 == "clean" ]]
then
	if [[ -d ${output_base} ]]
	then
		rm -rf ${output_base} && echo "deleted output directory ${output_base}."
	else 
		echo "${output_base} does not exist, nothing here for me to clean up."
	fi
	exit 0
fi

if [[ -z ${1+x} || ! -r $1 ]]
then
	usage
	exit 1
fi

widoco_jar=$1
if [[ ! -z ${2+x} ]]
then
	only_ont=$2
fi

ontology_root=${project_dir}/src/main/resources/ontology/
echo "(re)generating documentation for ontologies"
echo "   ontologies folder:  ${ontology_root}"
echo "   output folder    :  ${output_base}"

onts=(core message agreement modification content matching)
ext_onts=(buddy chat group hold review schema demo pogo bot)

rewrite_base="https://researchstudio-sat.github.io/webofneeds/ontologies"

function generate_for_ontology() {
	ont=$1
	is_ext=$2
	if [[ ! -z ${only_ont} && ${only_ont} != ${ont} ]]
	then
		# the user specified only_ont and it's not the current one. skip.
		echo "Only generating docs for ontology ${only_ont}. Skipping ${ont}"
		return
	fi
	if [[ ${is_ext} = true ]]
	then
		output_path="ext/${ont}"
		ont_src_dir="${ontology_root}/ext"
		ont_file_stem="won-ext-${ont}"
	else 
		output_path="${ont}"
		ont_src_dir="${ontology_root}"
		ont_file_stem="won-${ont}"
	fi
	ont_widoco_dir="${ont_src_dir}/${ont_file_stem}-widoco"
	ont_file_name="${ont_file_stem}.ttl"
	ont_file="${ont_src_dir}/${ont_file_name}"
	config_file="${ont_widoco_dir}/widoco.conf"
	if [[ -r ${config_file} ]]
	then
		config_file_opt="-confFile ${config_file}"
	fi
	mkdir -p "${output_path}"
	echo "generating documentation for '${ont}' ontology in subfolder '${output_path}'" 
	java -jar ${widoco_jar} -ontFile "${ont_file}" -outFolder "${output_path}" ${config_file_opt} \
	    -licensius -oops -rewriteAll -webVowl
	java -jar ${widoco_jar} -ontFile "${ont_file}" -outFolder "${output_path}" ${config_file_opt} \
		-crossRef -rewriteAll
	cp "${output_path}/index-en.html" "${output_path}/index.html"
	# rename ontology.json to ontology.jsonld because we host our ontologies on github
	# and their content-type system is extension based. (.jsonld -> application/ld+json)
	mv "${output_path}/ontology.json" "${output_path}/ontology.jsonld"
	# replace extension .json with .jsonld in the redirects file, if created
	sed -i -e 's/\.json/.jsonld/g' "${output_path}/.htaccess"
	for f in "${output_path}/index*.html"
	do
		sed -i -e 's/ontology\.json/ontology.jsonld/g' ${f}
	done
	for section_html_file in ${output_path}/sections/*.html
	do
		section_html_filename=${section_html_file##*/}
		section_md_filename=${section_html_filename%.html}.md
		section_md_file=${ont_widoco_dir}/${section_md_filename}
		section_name=${section_html_filename%.html}
		section_name=${section_name%-*}
		if [[ -r ${section_md_file} && -s  ${section_md_file} ]]
		then
			# md file is readable and non-empty
			echo -e "Replacing template text in ${section_html_filename} with content of ${section_md_file}"
			#perl -i -0pe "s@(?<=<span class=\"markdown\">)[^>]+(?=</span>)@\`cat ${section_md_file}\`@e" ${section_html_file}
		elif [[ ${section_name} == "description" || ${section_name} == "references" ]]
		then
			# we have no corresponding md file (or it is empty).
			# in case of description or references we can delete the whole section.
			echo "Removing section '${section_name}' as no markdown was provided for it in ${ont_widoco_dir}"
			rm ${section_html_file}
		elif [[ ${section_name} != "abstract" && ${section_name} != "crossref" ]]
		then
			# we have no corresponding md file (or it is empty).
			# just remove the placeholder text generated by widoco
			# Exceptions: 
			# - the abstract may have been populated from ontology properties so we exclude it here
			# - the crossref is entirely generated so don't touch it here
			echo -e "Removing template text in ${section_html_filename} as ${section_md_file} empty or missing"
			perl -i -0pe "s@(?<=<span class=\"markdown\">)[^>]+(?=</span>)@@" ${section_html_file}
		fi
	done
	if [[ -d ${ont_widoco_dir}/img ]]
	then
		echo "copying images folder"
		cp -a ${ont_widoco_dir}/img ${output_path}
	fi
}

mkdir -p ${output_base}
mkdir -p ${output_base}/ext
cd "${output_base}"

for ont in ${onts[@]} 
do
	generate_for_ontology ${ont} false
done

for ont in ${ext_onts[@]} 
do
	generate_for_ontology ${ont} true
done