package au.com.dius.pact.server

import au.com.dius.pact.core.model.{OptionalBody, Response}
import ch.qos.logback.classic.Level
import org.slf4j.{Logger, LoggerFactory}

import scala.collection.JavaConverters._

object ListServers {

  def apply(oldState: ServerState): Result = {
    val ports = oldState.keySet.filter(p => p.matches("\\d+")).mkString(", ")
    val paths = oldState.keySet.filter(p => !p.matches("\\d+")).map("\"" + _ + "\"").mkString(", ")
    val body = OptionalBody.body(("{\"ports\": [" + ports + "], \"paths\": [" + paths + "]}").getBytes)
    Result(new Response(200, Map("Content-Type" -> List("application/json").asJava).asJava, body), oldState)
  }
}

case class Result(response: Response, newState: ServerState)

case class Config(port: Int = 29999,
                  host: String = "localhost",
                  daemon: Boolean = false,
                  portLowerBound: Int = 20000,
                  portUpperBound: Int = 40000,
                  debug: Boolean = false,
                  pactVersion: Int = 2,
                  keystorePath: String = "",
                  keystorePassword: String = "",
                  sslPort : Int = 8443,
                  broker: String = "",
                  authToken: String = ""
                 )

object Server extends App {

  val parser = new scopt.OptionParser[Config]("pact-jvm-server") {
    arg[Int]("port") optional() action { (x, c) => c.copy(port = x) } text("port to run on (defaults to 29999)")
    help("help") text("prints this usage text")
    opt[String]('h', "host") action { (x, c) => c.copy(host = x) } text("host to bind to (defaults to localhost)")
    opt[Int]('l', "mock-port-lower") action { (x, c) => c.copy(portLowerBound = x) } text("lower bound to allocate mock ports (defaults to 20000)")
    opt[Int]('u', "mock-port-upper") action { (x, c) => c.copy(portUpperBound = x) } text("upper bound to allocate mock ports (defaults to 40000)")
    opt[Unit]('d', "daemon") action { (_, c) => c.copy(daemon = true) } text("run as a daemon process")
    opt[Unit]("debug") action { (_, c) => c.copy(debug = true) } text("run with debug logging")
    opt[Int]('v', "pact-version") action { (x, c) => c.copy(pactVersion = x) } text("pact version to generate for (2 or 3)")
    opt[String]('k', "keystore-path") action { (x, c) => c.copy(keystorePath = x) } text("Path to keystore")
    opt[String]('p', "keystore-password") action { (x, c) => c.copy(keystorePassword = x) } text("Keystore password")
    opt[Int]('s', "ssl-port") action { (x, c) => c.copy(sslPort = x) } text("Ssl port the mock server should run on. lower and upper bounds are ignored")
    opt[String]('b', "broker") action {(x, c) => c.copy(broker = x)} text("URL of broker where to publish contracts to")
    opt[String]('t', "token") action {(x, c) => c.copy(authToken = x)} text("Auth token for publishing the pact to broker")
  }

  parser.parse(args, Config()) match {
    case Some(config) =>
      val logger = LoggerFactory.getLogger(Logger.ROOT_LOGGER_NAME).asInstanceOf[ch.qos.logback.classic.Logger]
      if (config.debug) {
        logger.setLevel(Level.DEBUG)
      } else {
        logger.setLevel(Level.INFO)
      }
      val server = _root_.unfiltered.netty.Server.http(config.port, config.host)
        .handler(RequestHandler(new ServerStateStore(), config))

      if(!config.keystorePath.isEmpty) {
        println(s"Using keystore '${config.keystorePath}' for mock https server")
      }

      println(s"starting unfiltered app at ${config.host} on port ${config.port}")
      server.start()
      if (!config.daemon) {
        readLine("press enter to stop server:\n")
        server.stop()
      }

    case None =>
      parser.showUsage
  }
}
