package com.skytix.velocity.scheduler;

import com.skytix.velocity.MetricUtils;
import com.skytix.velocity.entities.VelocityTask;
import com.skytix.velocity.repository.TaskRepository;
import io.micrometer.core.instrument.MeterRegistry;
import lombok.extern.slf4j.Slf4j;
import org.apache.mesos.v1.Protos;
import org.apache.mesos.v1.scheduler.Protos.Event.Update;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.Collections;
import java.util.concurrent.Flow;
import java.util.concurrent.SubmissionPublisher;

@Slf4j
public class UpdateSubscriber implements Flow.Subscriber<Update> {
    private final TaskRepository<VelocityTask> mTaskRepository;
    private final SubmissionPublisher<TaskUpdateEvent> mEventUpdatePublisher;
    private final SchedulerRemoteProvider mRemote;
    private final TaskEventHandler mDefaultUpdateHandler;
    private final int mTaskRetryLimit;
    private final SubmissionPublisher<VelocityTask> mTaskPublisher;

    private final MeterRegistry mMeterRegistry;

    private Flow.Subscription mSubscription;

    public UpdateSubscriber(
            TaskRepository<VelocityTask> aTaskRepository,
            SubmissionPublisher<TaskUpdateEvent> aSubmissionPublisher,
            SubmissionPublisher<VelocityTask> aTaskPublisher,
            SchedulerRemoteProvider aRemote,
            TaskEventHandler aDefaultUpdateHandler,
            MeterRegistry aMeterRegistry,
            int aTaskRetryLimit) {
        mTaskRepository = aTaskRepository;
        mEventUpdatePublisher = aSubmissionPublisher;
        mRemote = aRemote;
        mDefaultUpdateHandler = aDefaultUpdateHandler;
        mTaskRetryLimit = aTaskRetryLimit;
        mTaskPublisher = aTaskPublisher;
        mMeterRegistry = aMeterRegistry;

        mEventUpdatePublisher.subscribe(new TaskEventUpdateSubscriber(aDefaultUpdateHandler));
    }

    @Override
    public void onSubscribe(Flow.Subscription subscription) {
        mSubscription = subscription;
        subscription.request(1);
    }

    @Override
    public void onNext(Update update) {

        try {
            final Protos.TaskStatus updateStatus = update.getStatus();
            final VelocityTask task = mTaskRepository.getTaskByTaskId(updateStatus.getTaskId().getValue());

            if (task != null) {
                mTaskRepository.updateTaskState(task, updateStatus.getState());
                // Send acknowledgement first to prevent handlers from delaying the potential release of resources.
                acknowledge(updateStatus);

                switch (updateStatus.getState()) {

                    case TASK_RUNNING:

                        if (!task.isRunning()) {
                            task.setRunning(true);
                            task.setStartTime(LocalDateTime.now());

                            MetricUtils.getTimerForTask(mMeterRegistry, "velocity.timer.scheduler.taskQueuedDuration", task)
                                    .record(Duration.between(task.getCreated(), LocalDateTime.now()));
                        }

                        break;

                    case TASK_FINISHED:

                        if (!task.isComplete()) {
                            task.setFinishTime(LocalDateTime.now());
                            recordTaskDuration(task);

                            MetricUtils.getCounterForTask(mMeterRegistry, "velocity.counter.scheduler.completedTasks", task)
                                    .increment();

                            mTaskRepository.completeTask(task);

                            suppressOffersIfIdle();
                        }

                        break;

                    case TASK_DROPPED:
                    case TASK_FAILED:
                    case TASK_ERROR:
                    case TASK_KILLED:
                    case TASK_GONE:
                    case TASK_GONE_BY_OPERATOR:
                    case TASK_LOST:
                        task.setFinishTime(LocalDateTime.now());
                        recordTaskDuration(task);

                        switch (updateStatus.getReason()) {
                            case REASON_CONTAINER_LAUNCH_FAILED:
                            case REASON_TASK_KILLED_DURING_LAUNCH:
                            case REASON_EXECUTOR_TERMINATED:
                            case REASON_GC_ERROR:
                            case REASON_INVALID_OFFERS:
                                // Retry the task since it may be ephemeral.

                                if (task.getTaskRetries() < mTaskRetryLimit) {
                                    log.debug(String.format("Task %s failed for reason: %s. Retrying...", updateStatus.getTaskId(), updateStatus.getReason()));

                                    MetricUtils.getCounterForTask(mMeterRegistry, "velocity.counter.scheduler.retriedTasks", task)
                                            .increment();

                                    mTaskRepository.completeTask(task);
                                    mTaskPublisher.submit(task);

                                    if (mTaskRepository.getNumQueuedTasks() > 0) { // If we just ran 1 task, the queue will be empty so we will want to revive the offers.
                                        mRemote.get().revive(Collections.emptyList());
                                    }

                                    return;
                                }

                                break;
                        }

                        failTask(updateStatus, task);
                        suppressOffersIfIdle();

                        break;
                }

                notifyUpdateHandler(update, task);

            } else {
                // We don't know about the task anymore so acknowledge the updates.
                acknowledge(updateStatus);

                if (mDefaultUpdateHandler != null) {
                    mDefaultUpdateHandler.onEvent(update);
                }

            }

        } catch (Exception aE) {
            log.error(aE.getMessage(), aE);

        } finally {
            mSubscription.request(1);
        }

    }

    private void failTask(Protos.TaskStatus aUpdateStatus, VelocityTask aTask) {

        MetricUtils.getCounterForTask(mMeterRegistry, "velocity.counter.scheduler.failedTasks", aTask)
                .increment();

        log.debug(String.format("Task %s failed for reason: (%s) %s.", aUpdateStatus.getTaskId(), aUpdateStatus.getReason(), aUpdateStatus.getMessage()));
        mTaskRepository.completeTask(aTask);
    }

    private void notifyUpdateHandler(Update update, VelocityTask aTask) {
        mEventUpdatePublisher.submit(
                TaskUpdateEvent.builder()
                        .event(update)
                        .task(aTask)
                        .build()
        );
    }

    private void suppressOffersIfIdle() {
        if (mTaskRepository.getNumQueuedTasks() == 0 && mTaskRepository.getNumActiveTasks() == 0) {
            log.debug("Scheduler is idle. Suppressing offers");
            mRemote.get().suppress(Collections.emptyList());
        }
    }

    private void recordTaskDuration(VelocityTask aTask) {
        MetricUtils.getTimerForTask(mMeterRegistry, "velocity.timer.scheduler.taskDuration", aTask).record(
                Duration.between(
                        aTask.getStartTime() != null ? aTask.getStartTime() : aTask.getCreated(),
                        aTask.getFinishTime() != null ? aTask.getFinishTime() : LocalDateTime.now())
        );

        MetricUtils.getTimerForTask(mMeterRegistry, "velocity.timer.scheduler.taskTotalDuration", aTask)
                .record(Duration.between(aTask.getCreated(),LocalDateTime.now()));
    }

    private void acknowledge(Protos.TaskStatus aStatus) {

        if (aStatus.hasUuid()) {
            mRemote.get().acknowledge(aStatus);
        }

    }

    @Override
    public void onError(Throwable throwable) {
        log.error(throwable.getMessage(), throwable);
    }

    @Override
    public void onComplete() {
        // Yay?
    }

}
