package br.com.jarch.jpa.api;

import br.com.jarch.jpa.util.ClauseFromUtils;
import br.com.jarch.jpa.util.ClauseGroupByUtils;
import br.com.jarch.jpa.util.ClauseOrderByUtils;
import br.com.jarch.jpa.util.ClauseWhereUtils;
import br.com.jarch.jpa.util.ValueWhereUtils;
import br.com.jarch.model.IBaseEntity;
import br.com.jarch.util.JpaUtils;
import br.com.jarch.util.LogUtils;
import br.com.jarch.util.type.EntityGraphType;
import org.hibernate.jpa.QueryHints;

import javax.persistence.NoResultException;
import javax.persistence.NonUniqueResultException;
import javax.persistence.TypedQuery;
import javax.persistence.metamodel.SingularAttribute;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

public final class CollectorJpaql<E extends IBaseEntity> {

    ClientJpaql<E> clientJpaql;

    CollectorJpaql(ClientJpaql<E> clientJpaql) {
        this.clientJpaql = clientJpaql;
    }

    public Object[] aggregate(IAggregate<? super E>... aggregateCommands) {
        if (aggregateCommands.length < 2) {
            return new Object[]{
                    getQuerySelect(Arrays
                            .stream(aggregateCommands)
                            .map(IAggregate::getCommand)
                            .collect(Collectors.joining(", ")), Object.class)
                            .getSingleResult()};
        }

        return getQuerySelect(Arrays
                .stream(aggregateCommands)
                .map(IAggregate::getCommand)
                .collect(Collectors.joining(", ")), Object[].class)
                .getSingleResult();
    }

    public <T> T aggregate(Class<T> clazz, IAggregate<? super E>... aggregateCommands) {
        return getQuerySelect("new " + clazz.getName() + "(" + Arrays.stream(aggregateCommands).map(IAggregate::getCommand).collect(Collectors.joining(", ")) + ")", clazz)
                .getSingleResult();
    }

    public Long count() {
        return getQueryCount().getSingleResult();
    }

    public boolean exists() {
        return count() > 0;
    }

    public List<E> list() {
        return getQuerySelect(alias(), classEntity()).getResultList();
    }

    public Set<E> set() {
        return list().stream().collect(Collectors.toSet());
    }

    @Deprecated
    public List<E> list(String entityGraph) {
        clientJpaql.entityGraph = entityGraph;
        return getQuerySelect(alias(), classEntity()).getResultList();
    }

    @Deprecated
    public List<E> list(JoinFetch<E>... joinFetches) {
        Arrays.stream(joinFetches).forEach(jf -> clientJpaql.joinFetchs.add(jf));
        return getQuerySelect(alias(), classEntity()).getResultList();
    }

    public <T> List<T> list(Class<T> clazz, String... fields) {
        return getQuerySelect("new " + clazz.getName() + "(" + Arrays.stream(fields).collect(Collectors.joining(", ")) + ")", clazz)
                .getResultList();
    }

    public <T> Set<T> set(Class<T> clazz, String... fields) {
        return list(clazz, fields).stream().collect(Collectors.toSet());
    }

    public <T> List<T> list(Class<T> clazz, SingularAttribute... singularAttributes) {
        return list(clazz, Arrays.stream(singularAttributes).map(SingularAttribute::getName).collect(Collectors.joining(", ")));
    }

    public <T> Set<T> set(Class<T> clazz, SingularAttribute... singularAttributes) {
        return list(clazz, singularAttributes).stream().collect(Collectors.toSet());
    }

    @Deprecated
    public <T> List<T> groupBy(Class<T> clazz) {
        return groupByList(clazz);
    }

    public <T> List<T> groupByList(Class<T> clazz) {
        return getQuerySelectGroup(clazz).getResultList();
    }

    public <T> Set<T> groupBySet(Class<T> clazz) {
        return groupByList(clazz).stream().collect(Collectors.toSet());
    }

    public E single() {
        return getQuerySelect(alias(), classEntity()).getSingleResult();
    }

    public Optional<E> singleOptional() {
        try {
            return Optional.of(single());
        } catch (NoResultException | NonUniqueResultException ex) {
            LogUtils.generate(ex);

            return Optional.empty();
        }
    }

    @Deprecated
    public E single(JoinFetch<E>... joinFetches) {
        Arrays.stream(joinFetches).forEach(jf -> clientJpaql.joinFetchs.add(jf));
        return getQuerySelect(alias(), classEntity()).getSingleResult();
    }

    @Deprecated
    public Optional<E> singleOptional(JoinFetch<E>... joinFetches) {
        try {
            return Optional.of(single(joinFetches));
        } catch (NoResultException | NonUniqueResultException ex) {
            return Optional.empty();
        }
    }

    private TypedQuery<Long> getQueryCount() {
        String jpaql = "SELECT COUNT(" + (clientJpaql.distinct ? "DISTINCT " : "") + JpaUtils.aliasEntity(classEntity()) + ") ";
        jpaql += ClauseFromUtils.generateClauseFrom(classEntity()) + " ";
        jpaql += ClauseWhereUtils.generateClauseWhere(classEntity(), clientJpaql.paramFieldValues);
        TypedQuery<Long> query = clientJpaql.getEntityManager().createQuery(jpaql, Long.class);

        setConfiguration(query);

        return query;
    }

    private <T> TypedQuery<T> getQuerySelect(String paramsSelect, Class<T> classReturn) {
        String jpaql = "SELECT " + (clientJpaql.distinct ? "DISTINCT " : "") + paramsSelect + " ";
        jpaql += ClauseFromUtils.generateClauseFrom(classEntity(), clientJpaql.joinFetchs) + " ";
        jpaql += ClauseWhereUtils.generateClauseWhere(classEntity(), clientJpaql.paramFieldValues) + " ";
//        jpaql += getClauseOrderBy();
        jpaql += ClauseOrderByUtils.generateClauseOrderBy(classEntity(), clientJpaql.orderBys);
        TypedQuery<T> query = clientJpaql.getEntityManager().createQuery(jpaql, classReturn);
        setConfiguration(query);
        return query;
    }

    private <T> TypedQuery<T> getQuerySelectGroup(Class<T> classReturn) {
        String jpaql = "SELECT new " + classReturn.getName() + "(" +
                clientJpaql.groupBys.stream().map(f -> alias() + "." + f).collect(Collectors.joining(", ")) +
                (clientJpaql.groupBys.isEmpty() || clientJpaql.groupBys.isEmpty() ? "" : ", ") +
                clientJpaql.aggregates.stream().map(IAggregate::getCommand).collect(Collectors.joining(", ")) +
                ") ";

        jpaql += ClauseFromUtils.generateClauseFrom(classEntity(), clientJpaql.joinFetchs) + " ";
        jpaql += ClauseWhereUtils.generateClauseWhere(classEntity(), clientJpaql.paramFieldValues) + " ";
        jpaql += ClauseGroupByUtils.generateClauseGroupBy(classEntity(), clientJpaql.groupBys) + " ";
        jpaql += ClauseOrderByUtils.generateClauseOrderBy(classEntity(), clientJpaql.orderBys);

        TypedQuery<T> query = clientJpaql.getEntityManager().createQuery(jpaql, classReturn);

        setConfiguration(query);

        return query;
    }

    private String alias() {
        return JpaUtils.aliasEntity(classEntity());
    }

    private Class<E> classEntity() {
        return clientJpaql.getClassEntity();
    }

    private void setConfiguration(TypedQuery typedQuery) {

        if (clientJpaql.entityGraph != null && !clientJpaql.entityGraph.isEmpty()) {

            if (clientJpaql.graphType == null || EntityGraphType.FETCH.equals(clientJpaql.graphType)) {
                typedQuery.setHint(QueryHints.HINT_FETCHGRAPH, clientJpaql.getEntityManager().getEntityGraph(clientJpaql.entityGraph));
            } else {
                typedQuery.setHint(QueryHints.HINT_LOADGRAPH, clientJpaql.getEntityManager().getEntityGraph(clientJpaql.entityGraph));
            }
        }

        if (clientJpaql.cacheable || JpaUtils.cached(clientJpaql.getClassEntity())) {
            typedQuery = JpaUtils.forceCache(typedQuery);
        }

        ValueWhereUtils.addValueParam(typedQuery, clientJpaql.paramFieldValues);

        if (clientJpaql.firstResult > 0) {
            typedQuery = typedQuery.setFirstResult(clientJpaql.firstResult);
        }
        if (clientJpaql.maxResults > 0) {
            typedQuery = typedQuery.setMaxResults(clientJpaql.maxResults);
        }
    }

//    private String getClauseOrderBy() {
//        if (clientJpaql.orderBys.isEmpty()) {
//            return "";
//        }
//
//        String aliasEntidade = JpaUtils.aliasEntity(classEntity());
//
//        return "ORDER BY " +
//                clientJpaql
//                        .orderBys
//                        .stream()
//                        .map(f -> aliasEntidade + "." + f.field + " " + (OrderType.ASC.equals(f.order) ? "" : "DESC"))
//                        .collect(Collectors.joining(", "));
//
//    }

}