/*
 * Copyright 2016 Objectos, Fábrica de Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package br.com.objectos.rio.tftp;

import java.util.List;

import br.com.objectos.rio.udp.Message;
import br.com.objectos.rio.udp.Packet;
import br.com.objectos.rio.udp.UdpException;

/**
 * @author marcio.endo@objectos.com.br (Marcio Endo)
 */
abstract class ReadSession {

  private final Packet packet;
  private final List<DataMessage> messageList;

  private int index;
  private TftpServer server;

  ReadSession(Packet packet, List<DataMessage> messageList) {
    this.packet = packet;
    this.messageList = messageList;
  }

  public static ReadSession of(Packet packet, OAckMessage ackMessage, List<DataMessage> messageList) {
    return ackMessage.isEmpty()
        ? new OptionsLessReadSession(packet, messageList)
        : new OptionsFullReadSession(packet, ackMessage, messageList);
  }

  public final boolean matches(Packet packet) {
    return this.packet.fromAddressOf(packet);
  }

  public final ReadSessionOnAck onAck(AckMessage message) {
    return ackProcessor().onMessage(message);
  }

  public final void start(TftpServer server) {
    server.register(this);
    this.server = server;
    executeStart(server);
  }

  abstract AckProcessor ackProcessor();

  abstract void executeStart(TftpServer server);

  final void reply(Message message) {
    try {
      packet.reply(message);
    } catch (UdpException e) {
      e.printStackTrace();
    }
  }

  final void sendCurrent() {
    DataMessage message = messageList.get(index);
    reply(message);
    server.onDataSent(message);
  }

  class AckProcessor {

    ReadSessionOnAck onMessage(AckMessage message) {
      index++;
      if (index < messageList.size()) {
        sendCurrent();
        return ReadSessionOnAck.noop();
      } else {
        return ReadSessionOnAck.unregister(ReadSession.this);
      }
    }

  }

}