/*
 * Copyright 2016 Objectos, Fábrica de Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package br.com.objectos.rio.tftp;

import java.io.IOException;
import java.nio.file.NoSuchFileException;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import br.com.objectos.rio.udp.Packet;
import br.com.objectos.rio.udp.Service;
import br.com.objectos.rio.udp.UdpException;

/**
 * @author marcio.endo@objectos.com.br (Marcio Endo)
 */
public class TftpServer extends AbstractTftpService {

  private final int port;
  private final DataProvider provider;

  private final List<TftpServerListener> listenerList = new ArrayList<>(2);
  private final List<ReadSession> readSessionList = new ArrayList<>(5);

  TftpServer(int port, DataProvider provider) {
    this.port = port;
    this.provider = provider;
  }

  public TftpServer addListener(TftpServerListener listener) {
    Objects.requireNonNull(listener);
    listenerList.add(listener);
    return this;
  }

  @Override
  public TftpServer start() throws UdpException {
    return (TftpServer) super.start();
  }

  @Override
  protected Service service() throws UdpException {
    return Service.builder()
        .forClass(getClass())
        .bindTo(port)
        .addIncomingPacketAction(this)
        .build();
  }

  @Override
  void handleAckRequest(Packet packet) {
    AckMessage message = packet.decode(AckMessage::read);
    onAckRequest(message);
  }

  @Override
  void handleDataRequest(Packet packet) {
    // noop
  }

  @Override
  void handleReadRequest(Packet packet) {
    ReadRequestMessage message = packet.decode(ReadRequestMessage::read);
    onReadRequest(message);
    String filename = message.filename();

    try {
      byte[] bytes = provider.open(filename);
      TftpFile file = new TftpFile(bytes.length);
      OAckMessage oackMessage = message.toOAckMessage(file);
      ReadOptions options = oackMessage.toReadOptions();
      List<DataMessage> messageList = options.toDataMessage(bytes);
      ReadSession session = ReadSession.of(packet, oackMessage, messageList);
      session.start(this);
    } catch (NoSuchFileException e) {
      try {
        ErrorMessage errorMessage = ErrorMessage.fileNotFound(filename);
        packet.reply(errorMessage);
        onErrorSent(errorMessage);
      } catch (UdpException udp) {
        udp.printStackTrace();
      }
    } catch (IOException e) {
      e.printStackTrace();
    }
  }

  @Override
  void handleError(Packet packet) {
    // noop;
  }

  void onDataSent(DataMessage message) {
    for (TftpServerListener listener : listenerList) {
      listener.onDataSent(message);
    }
  }

  void onOAckSent(OAckMessage message) {
    for (TftpServerListener listener : listenerList) {
      listener.onOAckSent(message);
    }
  }

  void register(ReadSession session) {
    readSessionList.add(session);
  }

  void unregister(ReadSession session) {
    readSessionList.remove(session);
  }

  private void onAckRequest(AckMessage message) {
    for (TftpServerListener listener : listenerList) {
      listener.onAckRequest(message);
    }
    readSessionList.stream()
        .map(s -> s.onAck(message))
        .collect(Collectors.toList())
        .forEach(action -> action.execute(this));
  }

  private void onReadRequest(ReadRequestMessage message) {
    for (TftpServerListener listener : listenerList) {
      listener.onReadRequest(message);
    }
  }

  private void onErrorSent(ErrorMessage message) {
    for (TftpServerListener listener : listenerList) {
      listener.onErrorSent(message);
    }
  }

}