/*
 * Copyright 2011 Objectos, Fábrica de Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package br.com.objectos.way.attach;

import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.util.UUID;

import javax.imageio.ImageIO;

import com.google.common.io.Files;

import org.imgscalr.Scalr;
import org.imgscalr.Scalr.Mode;

/**
 * @author marcio.endo@objectos.com.br (Marcio Endo)
 */
class AttachmentPageImpl implements AttachmentPage {

  private final String baseDir;
  private final UUID uuid;
  private final int number;

  public AttachmentPageImpl(Builder builder) {
    baseDir = builder.getBaseDir();
    uuid = builder.getUuid();
    number = builder.getNumber();
  }

  @Override
  public void delete() {
    File file = getFile();
    file.delete();
  }

  @Override
  public byte[] toByteArray(Size size) {
    byte[] bytes = new byte[] {};

    File file = getFile();
    if (file != null) {
      bytes = image(size, file);
    }

    return bytes;
  }

  @Override
  public byte[] toByteArray(SizeEnum enumValue) {
    Size size = enumValue.asSize();
    return toByteArray(size);
  }

  @Override
  public File getFile() {
    File dir = AttachmentIO.dir(baseDir, uuid);

    String name = getName();
    return new File(dir, name);
  }

  @Override
  public UUID getUuid() {
    return uuid;
  }

  @Override
  public int getNumber() {
    return number;
  }

  @Override
  public String getName() {
    return String.format("%s.%d", uuid, number);
  }

  private byte[] image(Size size, File file) {
    byte[] bytes = new byte[] {};

    try {
      if (!Size.isOriginal(size)) {

        BufferedImage img = ImageIO.read(file);
        BufferedImage thumb = Scalr.resize(
            img,
            size.getMethod().toScalr(),
            Mode.FIT_TO_WIDTH,
            size.getWidth(),
            Integer.MAX_VALUE);
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        ImageIO.write(thumb, "PNG", baos);
        bytes = baos.toByteArray();

      } else {

        bytes = Files.toByteArray(file);

      }

    } catch (IOException e) {
    }

    return bytes;
  }
}