/*
 * Copyright 2014 Objectos, Fábrica de Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package br.com.objectos.way.ui;

import static com.google.common.collect.Lists.newArrayList;

import java.nio.charset.Charset;
import java.util.List;
import java.util.Map;

import com.google.common.base.Optional;
import com.google.common.collect.Lists;
import com.google.inject.Inject;
import com.google.inject.Injector;
import com.google.sitebricks.headless.Reply;
import com.google.sitebricks.headless.Request;
import com.google.sitebricks.routing.Action;

/**
 * @author marcio.endo@objectos.com.br (Marcio Endo)
 */
public abstract class AbstractPage implements Action {

  @Inject
  private Injector injector;

  private Request theRequest;
  private Map<String, String> urlMap;
  private PageRequest request;

  @Override
  public final boolean shouldCall(Request theRequest) {
    this.theRequest = theRequest;
    return true;
  }

  @Override
  public final Object call(Request theRequest, Object page, Map<String, String> map) {
    urlMap = map;

    PageRequest previousRequest = previousRequest();
    request = request(previousRequest);

    Optional<Reply<?>> reply = request
        .toPojo()
        .shouldUI();

    if (!reply.isPresent()) {
      reply = ui(request)
          .toPojo()
          .reply();
    }

    return reply.get();
  }

  protected Class<? extends AbstractPage> after() {
    return null;
  }

  protected abstract PageRequest request(PageRequest request);

  protected abstract PageUI ui(PageRequest request);

  protected PageRequestBuilder newRequestBuilder(PageRequest request) {
    return injector.getInstance(PageRequestBuilderPojo.class).after(request);
  }

  protected PageUIBuilder newUIBuilder(PageRequest request) {
    Class<? extends AbstractPage> previousPageClass = after();
    return injector.getInstance(PageUIBuilderPojo.class).with(request).after(previousPageClass);
  }

  protected ParamBuilder param(String key) {
    return new ReqParamBuilder(key);
  }

  protected ParamBuilder urlPart(String key) {
    return new UrlParamBuilder(key);
  }

  AbstractPage withUrlMap(Map<String, String> urlMap) {
    this.urlMap = urlMap;
    return this;
  }

  private PageRequest previousRequest() {
    List<Class<? extends AbstractPage>> previousClasses;
    previousClasses = newArrayList();

    Class<? extends AbstractPage> previousPageClass;
    previousPageClass = after();

    while (previousPageClass != null) {
      previousClasses.add(previousPageClass);

      AbstractPage page;
      page = injector.getInstance(previousPageClass);

      previousPageClass = page.after();
    }

    List<PageRequest> pages;
    pages = newArrayList();

    PageRequest request;
    request = injector.getInstance(PageRequestPojo.class);

    List<Class<? extends AbstractPage>> reversed = Lists.reverse(previousClasses);
    for (Class<? extends AbstractPage> pageClass : reversed) {
      AbstractPage page;
      page = injector.getInstance(pageClass);

      request = page
          .withUrlMap(urlMap)
          .request(request);

      pages.add(request);
    }

    return request.toPojo()
        .addAll(pages);
  }

  protected abstract class ParamBuilder {

    private final String key;

    private final ParameterWrapper wrapper;

    public ParamBuilder(String key) {
      this.key = key;
      this.wrapper = newWrapper();
    }

    public String get() {
      return wrapper.get(key);
    }

    public int getInt() {
      return wrapper.getInt(key);
    }

    public long getLong() {
      return wrapper.getLong(key);
    }

    public final String decodeAndGet(Charset charset) {
      return wrapper.decodeAndGet(key, charset);
    }

    abstract ParameterWrapper newWrapper();

  }

  private class ReqParamBuilder extends ParamBuilder {

    public ReqParamBuilder(String key) {
      super(key);
    }

    @Override
    ParameterWrapper newWrapper() {
      return ParameterWrapper.of(theRequest);
    }

  }

  private class UrlParamBuilder extends ParamBuilder {

    public UrlParamBuilder(String key) {
      super(key);
    }

    @Override
    ParameterWrapper newWrapper() {
      return ParameterWrapper.of(urlMap);
    }

  }

}