/*
 * Copyright 2014 Objectos, Fábrica de Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package br.com.objectos.way.ui;

import static com.google.common.collect.Sets.newHashSet;

import java.util.Optional;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;

import com.google.inject.Inject;
import com.google.inject.Provider;
import com.google.sitebricks.headless.Request;

/**
 * @author marcio.endo@objectos.com.br (Marcio Endo)
 */
class PageRequestBuilderPojo
    implements
    PageRequestBuilder {

  private final Provider<Request> requests;
  private final Provider<HttpServletRequest> servletRequests;

  private final Set<Method> methods = newHashSet();
  private final Set<PageCondition> conditions = newHashSet();

  @Inject
  public PageRequestBuilderPojo(Provider<Request> requests,
                                Provider<HttpServletRequest> servletRequests) {
    this.requests = requests;
    this.servletRequests = servletRequests;
  }

  @Override
  public <T> PageRequestBuilder put(Class<T> key, Optional<? extends T> optional) {
    store(key, optional.orElse(null));
    return this;
  }

  @Override
  public <T> PageRequestBuilder put(Class<T> key, com.google.common.base.Optional<? extends T> optional) {
    store(key, optional.orNull());
    return this;
  }

  @Override
  public <T> PageRequestBuilder put(Class<T> key, T value) {
    store(key, value);
    return this;
  }

  @Override
  public RespondToBuilder respondTo() {
    return new RespondToBuilderPojo();
  }

  PageRequestBuilderPojo after(PageRequest request) {
    request
        .toPojo()
        .addAllTo(conditions);
    return this;
  }

  Method getMethod() {
    Request request = requests.get();
    return Method.parse(request);
  }

  Provider<HttpServletRequest> getServletRequests() {
    return servletRequests;
  }

  Set<Method> getMethods() {
    return methods;
  }

  Set<PageCondition> getConditions() {
    return conditions;
  }

  private PageRequest build() {
    return new PageRequestPojo(this);
  }

  private void store(Class<?> key, Object value) {
    HttpServletRequest request = servletRequests.get();
    request.setAttribute(key.getName(), value);
  }

  private class RespondToBuilderPojo
      implements
      PageRequestBuilder.RespondToBuilder,
      PageRequestBuilder.GetBuilder,
      PageRequestBuilder.PostBuilder {

    @Override
    public GetBuilder get() {
      methods.add(Method.GET);
      return this;
    }

    @Override
    public PostBuilder post() {
      methods.add(Method.POST);
      return this;
    }

    @Override
    public SayBuilder<PageRequest> always() {
      PageCondition condition = PageCondition.trueValue(conditions);
      return new SayBuilderPageRequest(condition);
    }

    @Override
    public SayBuilder<PageRequest> otherwise() {
      PageCondition condition = PageCondition.trueValue(conditions);
      return new SayBuilderPageRequest(condition);
    }

    @Override
    public WhenBuilder when(Class<?> key) {
      return new WhenBuilderPojo(key);
    }

  }

  private class SayBuilderPageRequest
      implements
      PageRequestBuilder.SayBuilder<PageRequest> {

    private final PageCondition condition;

    public SayBuilderPageRequest(PageCondition condition) {
      this.condition = condition;
    }

    @Override
    public PageRequest sayOk() {
      condition.sayOk();
      return build();
    }

    @Override
    public PageRequest sayNotFound() {
      condition.sayNotFound();
      return build();
    }

  }

  private class WhenBuilderPojo
      implements
      PageRequestBuilder.WhenBuilder {

    private final Class<?> key;

    public WhenBuilderPojo(Class<?> key) {
      this.key = key;
    }

    @Override
    public SayBuilder<ConditionBuilder> notPresent() {
      PageCondition condition = PageCondition.requestNotPresent(conditions, key);
      return new SayBuilderConditionBuilder(condition);
    }

  }

  private class SayBuilderConditionBuilder
      implements
      PageRequestBuilder.SayBuilder<ConditionBuilder> {

    private final PageCondition condition;

    public SayBuilderConditionBuilder(PageCondition condition) {
      this.condition = condition;
    }

    @Override
    public ConditionBuilder sayOk() {
      condition.sayOk();
      return new RespondToBuilderPojo();
    }

    @Override
    public ConditionBuilder sayNotFound() {
      condition.sayNotFound();
      return new RespondToBuilderPojo();
    }

  }

}