/*
 * Copyright 2014 Objectos, Fábrica de Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package br.com.objectos.way.ui;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.nio.charset.Charset;
import java.util.Map;

import com.google.sitebricks.headless.Request;

/**
 * @author marcio.endo@objectos.com.br (Marcio Endo)
 */
abstract class ParameterWrapper {

  ParameterWrapper() {
  }

  public static ParameterWrapper of(Map<String, String> urlMap) {
    return new UrlMapWrapper(urlMap);
  }

  public static ParameterWrapper of(Request request) {
    return new RequestWrapper(request);
  }

  public abstract String get(String key);

  public int getInt(String key) {
    try {
      String text = get(key);
      return Integer.parseInt(text);
    } catch (NumberFormatException e) {
      return 0;
    }
  }

  public long getLong(String key) {
    try {
      String text = get(key);
      return Long.parseLong(text);
    } catch (NumberFormatException e) {
      return 0l;
    }
  }

  public final String decodeAndGet(String key, Charset charset) {
    try {
      return URLDecoder.decode(get(key), charset.name());
    } catch (UnsupportedEncodingException e) {
      return "";
    }
  }

  private static class UrlMapWrapper extends ParameterWrapper {

    private final Map<String, String> urlMap;

    public UrlMapWrapper(Map<String, String> urlMap) {
      this.urlMap = urlMap;
    }

    @Override
    public String get(String key) {
      return urlMap.get(key);
    }

  }

  private static class RequestWrapper extends ParameterWrapper {

    private final Request request;

    public RequestWrapper(Request request) {
      this.request = request;
    }

    @Override
    public String get(String key) {
      return request.param(key);
    }

  }

}