/*
 * Copyright 2014 Objectos, Fábrica de Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package br.com.objectos.way.ui;

import java.io.IOException;
import java.io.InputStream;

import javax.servlet.http.HttpServletRequest;

import com.google.inject.Inject;
import com.google.inject.Injector;
import com.google.sitebricks.client.transport.Json;
import com.google.sitebricks.headless.Reply;

import org.codehaus.jackson.JsonGenerator;
import org.codehaus.jackson.JsonProcessingException;
import org.codehaus.jackson.map.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author marcio.endo@objectos.com.br (Marcio Endo)
 */
public class WayUI {

  private static final Logger logger = LoggerFactory.getLogger(WayUI.class);

  private final Injector injector;

  @Inject
  WayUI(Injector injector) {
    this.injector = injector;
  }

  public static ToJsonHelper toJsonHelper(JsonGenerator jgen) {
    return new ToJsonHelper(jgen);
  }

  public static UserInputConfigure userInputBuilder() {
    return new UserInputConfigurePojo();
  }

  public <T extends UserInputDelete> Reply<?> doDelete(Class<T> type) {
    return new DoDelete(type).reply();
  }

  public <T extends UserInputPost> Reply<?> doPost(Class<T> type) {
    return new DoPost(type).reply();
  }

  public <T extends UserInputPut> Reply<?> doPut(Class<T> type) {
    return new DoPut(type).reply();
  }

  private class DoDelete extends DoAction<UserInputDelete> {

    public DoDelete(Class<? extends UserInputDelete> type) {
      super(type);
    }

    @Override
    UserInput execute(UserInputDelete pojo) {
      return pojo.doDelete();
    }

  }

  private class DoPost extends DoAction<UserInputPost> {

    public DoPost(Class<? extends UserInputPost> type) {
      super(type);
    }

    @Override
    UserInput execute(UserInputPost pojo) {
      return pojo.doPost();
    }

  }

  private class DoPut extends DoAction<UserInputPut> {

    public DoPut(Class<? extends UserInputPut> type) {
      super(type);
    }

    @Override
    UserInput execute(UserInputPut pojo) {
      return pojo.doPut();
    }

  }

  private abstract class DoAction<T extends UserInputVerb> {

    private final Class<? extends T> type;

    public DoAction(Class<? extends T> type) {
      this.type = type;
    }

    public Reply<?> reply() {
      try {
        T pojo = injector.getInstance(type);
        return tryToReply(pojo);
      } catch (Exception e) {
        logger.error("Could not doAction()", e);
        return Reply.saying().error();
      }
    }

    abstract UserInput execute(T pojo);

    private Reply<?> tryToReply(T pojo) throws JsonProcessingException, IOException {
      InputStream src = injector.getProvider(HttpServletRequest.class)
          .get()
          .getInputStream();

      injector.getInstance(ObjectMapper.class)
          .readerForUpdating(pojo)
          .readValue(src);

      UserInput json = execute(pojo);

      return Reply.with(json).as(Json.class);
    }

  }

}