/*
 * Copyright 2013 Objectos, Fábrica de Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package br.com.objectos.way.ui.builder;

import static com.google.common.collect.Maps.newLinkedHashMap;

import java.util.Map;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.base.Strings;
import com.google.common.collect.Maps;

/**
 * @author marcio.endo@objectos.com.br (Marcio Endo)
 */
abstract class AbstractHtmlElement<E extends HtmlElement<E>>
    extends Component
    implements HtmlElement<E> {

  private final String tagName;

  private final Map<String, String> attributes = newLinkedHashMap();

  private final TagWriter tagWriter;

  private int depth = 0;

  private String text;

  public AbstractHtmlElement(String tagName) {
    this.tagName = tagName;
    this.tagWriter = selfClosing() ? new SelfClosingTag() : new Tag();
  }

  @Override
  public int getDepth() {
    return depth;
  }

  @Override
  public UIObject nextDepth() {
    this.depth++;
    return this;
  }

  @Override
  public E add(UIObject child) {
    super.addChild(child);
    return self();
  }

  @Override
  public E id(String id) {
    this.attributes.put("id", id);
    return self();
  }

  @Override
  public E styleClass(String styleClass) {
    this.attributes.put("class", styleClass);
    return self();
  }

  @Override
  public E text(String text) {
    this.text = text;
    return self();
  }

  @Override
  public E end() {
    return self();
  }

  protected void attr(String name, String value) {
    this.attributes.put(name, value);
  }

  @Override
  protected final void writeStartTag() {
    tagWriter.start();
  }

  @Override
  protected final void writeEndTag() {
    tagWriter.end();
  }

  abstract E self();

  boolean selfClosing() {
    return false;
  }

  private final String getTagName() {
    return tagName;
  }

  protected Map<String, String> getAttributes() {
    return attributes;
  }

  protected static enum Quoter implements Function<String, String> {

    INSTANCE;

    @Override
    public String apply(String input) {
      return String.format("\"%s\"", input);
    }

  }

  private abstract class TagWriter {
    void start() {
      writeIndent();
      write(String.format("<%s", getTagName()));

      if (!attributes.isEmpty()) {
        Map<String, String> quoted = Maps.transformValues(attributes, Quoter.INSTANCE);
        String res = Joiner.on(" ").withKeyValueSeparator("=").join(quoted);
        write(" " + res);
      }
    }

    void end() {
      if (getDepth() != 0) {
        writeNewLine();
      }
    }
  }

  private class Tag extends TagWriter {
    @Override
    void start() {
      super.start();

      write(">");

      String text = Strings.nullToEmpty(AbstractHtmlElement.this.text);
      write(text);

      if (hasChildren()) {
        writeNewLine();
      }
    }

    @Override
    public void end() {
      if (hasChildren()) {
        writeIndent();
      }

      write(String.format("</%s>", getTagName()));

      super.end();
    }
  }

  private class SelfClosingTag extends TagWriter {
    @Override
    void start() {
      super.start();
    }

    @Override
    public final void end() {
      write(" />");
      super.end();
    }
  }

}