/*
 * Copyright 2013 Objectos, Fábrica de Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package br.com.objectos.way.ui.form;

import static com.google.common.collect.Lists.newArrayList;

import java.util.List;

import br.com.objectos.way.ui.form.FormResponse.Error;
import br.com.objectos.way.ui.json.EntityJson;

/**
 * @author marcio.endo@objectos.com.br (Marcio Endo)
 */
public abstract class AbstractValidator<T extends EntityJson> {

  private final List<Error> errors = newArrayList();

  protected abstract void validate(T pojo);

  protected WhenBuilder when(boolean condition) {
    return new WhenBuilder(condition);
  }

  protected MessageBuilder<T> addMessage(String template, Object... args) {
    String message = String.format(template, args);
    return addMessage(message);
  }

  protected MessageBuilder<T> addMessage(final String message) {
    return new MessageBuilder<T>() {
      @Override
      public AbstractValidator<T> toField(String field) {
        errors.add(Errors.toField(field, message));
        return AbstractValidator.this;
      }

      @Override
      public AbstractValidator<T> toForm() {
        errors.add(Errors.toForm(message));
        return AbstractValidator.this;
      }
    };
  }

  List<Error> getErrors() {
    return errors;
  }

  protected static interface MessageBuilder<T extends EntityJson> {
    AbstractValidator<T> toField(String field);
    AbstractValidator<T> toForm();
  }

  protected class WhenBuilder {

    private final boolean condition;

    public WhenBuilder(boolean condition) {
      this.condition = condition;
    }

    public MessageBuilder<T> addMessage(String messageTemplate, Object... args) {
      return condition ?
          AbstractValidator.this.addMessage(messageTemplate, args) : new NoopMessageBuilder();
    }

    public MessageBuilder<T> addMessage(final String message) {
      return condition ?
          AbstractValidator.this.addMessage(message) : new NoopMessageBuilder();
    }

  }

  private class NoopMessageBuilder implements MessageBuilder<T> {

    @Override
    public AbstractValidator<T> toField(String field) {
      return AbstractValidator.this;
    }

    @Override
    public AbstractValidator<T> toForm() {
      return AbstractValidator.this;
    }

  }

}