/*
 * Copyright 2011 Objectos, Fábrica de Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package br.com.objectos.way.ui.form;

import static com.google.common.collect.Lists.newArrayList;

import java.util.List;

import br.com.objectos.way.ui.Context;
import br.com.objectos.way.ui.form.FormResponse.Error;

import com.google.sitebricks.client.transport.Json;
import com.google.sitebricks.headless.Reply;

/**
 * @author marcio.endo@objectos.com.br (Marcio Endo)
 */
public class FormResponseBuilder {

  private Context context;

  private final List<Error> errors = newArrayList();

  private final List<Error> formErrors = newArrayList();

  private String redirectUrl;

  public FormResponse build() {
    return new Impl(this);
  }

  public FormResponse get() {
    return build();
  }

  public Reply<?> reply() {
    FormResponse response = get();
    return Reply.with(response).as(Json.class);
  }

  public FormResponseBuilder withContext(Context context) {
    this.context = context;
    return this;
  }

  public FormResponseWhenBuilder when(boolean condition) {
    return new WhenBuilder(condition);
  }

  public FormResponseMessageBuilder addMessage(String template, Object... args) {
    String message = String.format(template, args);
    return addMessage(message);
  }

  public FormResponseMessageBuilder addMessage(String message) {
    return new ActualMessageBuilder(message);
  }

  public FormResponseBuilder redirectTo(String url) {
    this.redirectUrl = url;
    return this;
  }

  private class WhenBuilder implements FormResponseWhenBuilder {

    private final FormResponseBuilder outer = FormResponseBuilder.this;

    private final boolean condition;

    public WhenBuilder(boolean condition) {
      this.condition = condition;
    }

    @Override
    public FormResponseMessageBuilder addMessage(String messageTemplate, Object... args) {
      return condition ? outer.addMessage(messageTemplate, args) : new NoopMessageBuilder();
    }

    @Override
    public FormResponseMessageBuilder addMessage(final String message) {
      return condition ? outer.addMessage(message) : new NoopMessageBuilder();
    }

  }

  private class ActualMessageBuilder implements FormResponseMessageBuilder {

    private final String message;

    public ActualMessageBuilder(String message) {
      this.message = message;
    }

    @Override
    public FormResponseBuilder toField(String field) {
      errors.add(Errors.toField(field, message));

      return FormResponseBuilder.this;
    }

    @Override
    public FormResponseBuilder toForm() {
      errors.add(Errors.toForm(message));

      return FormResponseBuilder.this;
    }

  }

  private class NoopMessageBuilder implements FormResponseMessageBuilder {

    @Override
    public FormResponseBuilder toField(String string) {
      return FormResponseBuilder.this;
    }

    @Override
    public FormResponseBuilder toForm() {
      return FormResponseBuilder.this;
    }

  }

  private static class Impl implements FormResponse {

    private final Context context;
    private final List<Error> errors;
    private final List<Error> formErrors;
    private final String redirectUrl;

    public Impl(FormResponseBuilder builder) {
      this.context = builder.context;
      this.errors = builder.errors;
      this.formErrors = builder.formErrors;
      this.redirectUrl = builder.redirectUrl;
    }

    @Override
    public Context getContext() {
      return context;
    }

    @Override
    public List<Error> getErrors() {
      return errors;
    }

    @Override
    public List<Error> getFormErrors() {
      return formErrors;
    }

    @Override
    public String getRedirectUrl() {
      return redirectUrl;
    }

    @Override
    public boolean isValid() {
      return errors.isEmpty() && formErrors.isEmpty();
    }

  }

}