/*
 * Copyright 2013 Objectos, Fábrica de Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package br.com.objectos.way.upload.async;

import java.io.File;

import com.google.common.base.Objects;
import com.google.common.base.Preconditions;

import org.joda.time.DateTime;

/**
 * @author marcio.endo@objectos.com.br (Marcio Endo)
 */
public class AsyncFileKey {

  private final File dir;
  private final String name;

  private AsyncFileKey(File dir, String name) {
    this.dir = dir;
    this.name = name;
  }

  public static AsyncFileKey withExt(File dir, String ext) {
    Preconditions.checkArgument(dir.isDirectory(), "dir must be a directory");

    DateTime now = new DateTime();
    String filename = String.format("%s.%s", now.toString("yyyy-MM-dd-hhmmss"), ext);

    return new AsyncFileKey(dir, filename);
  }

  public static AsyncFileKey withName(File dir, String filename) {
    Preconditions.checkArgument(dir.isDirectory(), "dir must be a directory");
    return new AsyncFileKey(dir, filename);
  }

  public File getDir() {
    return dir;
  }

  public String getName() {
    return name;
  }

  public File toFile() {
    return new File(dir, name);
  }

  @Override
  public final int hashCode() {
    return Objects.hashCode(dir, name);
  }

  @Override
  public final boolean equals(final Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj == null) {
      return false;
    }
    if (obj instanceof AsyncFileKey) {
      final AsyncFileKey that = (AsyncFileKey) obj;
      return Objects.equal(this.dir, that.dir)
          && Objects.equal(this.name, that.name);
    } else {
      return false;
    }
  }

}