/*
 * Copyright 2013 Objectos, Fábrica de Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package br.com.objectos.way.upload.async;

import static com.google.common.collect.Lists.newArrayList;

import java.io.File;
import java.io.IOException;
import java.util.Date;
import java.util.List;

import com.google.common.base.Charsets;
import com.google.common.base.Joiner;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableList;
import com.google.common.io.Files;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author marcio.endo@objectos.com.br (Marcio Endo)
 */
public class AsyncFileLog {

  private static final String SEPARATOR = System.getProperty("line.separator");
  private static final Logger logger = LoggerFactory.getLogger(AsyncFileLog.class);

  private final AsyncFileKey fileKey;
  private final List<String> log;

  private AsyncFileLog(AsyncFileKey fileKey, List<String> log) {
    this.fileKey = fileKey;
    this.log = log;
  }

  public static AsyncFileLog logOf(AsyncFileKey fileKey) {
    List<String> log = newArrayList();
    return new AsyncFileLog(fileKey, log);
  }

  public static AsyncFileLog readLogOf(AsyncFileKey fileKey) {
    File logFile = logFileOf(fileKey);
    List<String> log;
    try {
      log = Files.readLines(logFile, Charsets.UTF_8);
    } catch (IOException e) {
      log = ImmutableList.of();
    }
    return new AsyncFileLog(fileKey, log);
  }

  public void debug(String format, Object... args) {
    String msg = msg("DEBUG", format, args);
    logger.debug(msg);
  }

  public void info(String format, Object... args) {
    String msg = msg("INFO", format, args);
    logger.info(msg);
  }

  public void error(String format, Object... args) {
    String msg = msg("ERROR", format, args);
    logger.error(msg);
  }

  public void error(String format, Throwable e, Object... args) {
    String msg = msg("ERROR", format, args);
    logger.error(msg, e);
    String trace = Throwables.getStackTraceAsString(e);
    log.add(trace);
  }

  public List<String> getLog() {
    return log;
  }

  public synchronized void flush() {
    try {
      String text = Joiner.on(SEPARATOR).join(log);
      File file = logFileOf(fileKey);
      Files.write(text, file, Charsets.UTF_8);
    } catch (IOException e) {

    }
  }

  @Override
  public String toString() {
    return log.toString();
  }

  private String msg(String level, String format, Object... args) {
    String msg = String.format(format, args);
    logAndFlush(level, msg);
    return msg;
  }

  private void logAndFlush(String level, String message) {
    String prefix = String.format("[%1$s] %2$tY-%2$tm-%2$td %2$tH:%2$tM:%2$tS", level, new Date());
    String msg = prefix + " " + message;

    log.add(msg);
    flushIfNecessary();
  }

  private void flushIfNecessary() {
    int size = log.size();
    if (size % 5 == 0) {
      flush();
    }
  }

  private static File logFileOf(AsyncFileKey fileKey) {
    File dir = fileKey.getDir();
    String name = fileKey.getName();
    return new File(dir, name + ".log");
  }

}