/*
 * Copyright 2012 Objectos, Fábrica de Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package br.com.objectos.way.ui;

import static com.google.common.collect.Lists.newArrayList;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import br.com.objectos.way.relational.Page;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;

/**
 * @author marcio.endo@objectos.com.br (Marcio Endo)
 */
public class PageList<T> {

  private final boolean empty;

  private final List<T> rows;

  private final Pager pager;

  public PageList(List<T> rows) {
    this.empty = rows.isEmpty();
    this.rows = rows;
    this.pager = new PagerJson(rows);
  }

  public PageList(List<T> rows, Pager pager) {
    this.empty = rows.isEmpty();
    this.rows = rows;
    this.pager = pager;
  }

  public static <T> PageList<T> of() {
    List<T> list = ImmutableList.of();
    return new PageList<T>(list);
  }

  public static <T> PageList<T> of(Iterable<T> rows) {
    List<T> list = ImmutableList.copyOf(rows);
    return new PageList<T>(list);
  }

  public PageList<T> filter(Predicate<T> predicate) {
    List<T> filtered = FluentIterable.from(rows)
        .filter(predicate)
        .toList();
    return PageList.of(filtered);
  }

  public PageList<T> page(RequestWrapper wrapper) {
    Page page = wrapper.getPage();
    List<T> limit = page.limit(rows);
    Pager nextPager = pager.to(page);
    return new PageList<T>(limit, nextPager);
  }

  public PageList<T> sort(Comparator<? super T> comparator) {
    List<T> rows = getRows();
    ArrayList<T> mutable = newArrayList(rows);
    Collections.sort(mutable, comparator);
    return new PageList<T>(mutable, pager);
  }

  public <E> PageList<E> transform(Function<T, E> function) {
    List<E> lazy = Lists.transform(rows, function);
    List<E> rows = ImmutableList.copyOf(lazy);
    return new PageList<E>(rows, pager);
  }

  public Context toContext(String var) {
    return Context.of()
        .put(var, this);
  }

  public boolean isEmpty() {
    return empty;
  }

  public List<T> getRows() {
    return rows;
  }

  public Pager getPager() {
    return pager;
  }

  public int size() {
    return rows.size();
  }

}