/*
 * Copyright 2011 Objectos, Fábrica de Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package br.com.objectos.way.attach;

import java.io.File;
import java.util.UUID;

import br.com.objectos.way.upload.AbstractUploadRedirect;
import br.com.objectos.way.upload.AbstractUploadSingleAction;
import br.com.objectos.way.upload.UploadRedirector;
import br.com.objectos.way.upload.UploadedRequest;
import br.com.objectos.way.upload.Uploads;

import com.google.inject.Inject;
import com.google.sitebricks.headless.Reply;

/**
 * @author marcio.endo@objectos.com.br (Marcio Endo)
 */
class AttachmentsGuice implements Attachments {

  private final Uploads uploads;

  private final AttachmentPages pages;

  @Inject
  public AttachmentsGuice(Uploads uploads, AttachmentPages pages) {
    this.uploads = uploads;
    this.pages = pages;
  }

  @Override
  public FirstFromRequest getFirstFromRequest() {
    return new Builder();
  }

  private class Builder implements FirstFromRequest, AndSaveTo {

    private File baseDir;

    @Override
    public AndSaveTo andSaveTo(File baseDir) {
      this.baseDir = baseDir;
      return this;
    }

    @Override
    public <T> Execute<T> execute(AttachmentAction<T> action) {
      return new GenericBuilder<T>(baseDir, action);
    }

  }

  private class GenericBuilder<T> implements Execute<T>, RedirectWith<T> {

    private final File baseDir;

    private final AttachmentAction<T> action;

    private AttachmentRedirect<T> redirect;

    public GenericBuilder(File baseDir, AttachmentAction<T> action) {
      this.baseDir = baseDir;
      this.action = action;
    }

    @Override
    public RedirectWith<T> redirectWith(AttachmentRedirect<T> redirect) {
      this.redirect = redirect;
      return this;
    }

    @Override
    public Reply<?> reply() {
      UUID uuid = UUID.randomUUID();
      File file = AttachmentIO.fileOf(baseDir, uuid);

      br.com.objectos.way.upload.Execute<T> execute = uploads.getFirstFromRequest()
          .andSaveToFile(file)
          .execute(new AttachUploadAction<T>(baseDir, action));

      if (redirect != null) {
        return execute
            .redirectWith(new AttachRedirect<T>(redirect))
            .reply();
      } else {
        return execute
            .reply();
      }
    }

  }

  private class AttachUploadAction<T> extends AbstractUploadSingleAction<T> {

    private final File baseDir;

    private final AttachmentAction<T> action;

    public AttachUploadAction(File baseDir, AttachmentAction<T> action) {
      this.baseDir = baseDir;
      this.action = action;
    }

    @Override
    public T execute(File file, UploadedRequest request) throws Exception {
      Attachment attach = new ProtoBuilder(baseDir, file, request).build();
      return action.execute(attach, file, request);
    }

  }

  private class AttachRedirect<T> extends AbstractUploadRedirect<T> {

    private final AttachmentRedirect<T> redirect;

    public AttachRedirect(AttachmentRedirect<T> redirect) {
      this.redirect = redirect;
    }

    @Override
    public String onSuccess(UploadRedirector redirector, T pojo) {
      return redirect.onSuccess(redirector, pojo);
    }

    @Override
    public String onError(UploadRedirector redirector) {
      return redirect.onError(redirector);
    }

  }

  private class ProtoBuilder implements Attachment.Builder {

    private final File baseDir;

    private final File file;

    private final UploadedRequest request;

    public ProtoBuilder(File baseDir, File file, UploadedRequest request) {
      this.baseDir = baseDir;
      this.file = file;
      this.request = request;
    }

    @Override
    public Attachment build() {
      ProtoAttachment attach = new ProtoAttachment(this);
      return pages.addPagesTo(attach);
    }

    @Override
    public File getBaseDir() {
      return baseDir;
    }

    @Override
    public UUID getUuid() {
      String name = file.getName();
      return UUID.fromString(name);
    }

    @Override
    public Mime getMime() {
      String name = getName();
      return Mime.valueOfContentType(name);
    }

    @Override
    public String getName() {
      return request.getFileName();
    }

  }

}