/*
 * Copyright 2014 Objectos, Fábrica de Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package br.com.objectos.way.ui;

import com.google.common.base.Strings;
import com.google.inject.Inject;
import com.google.inject.Injector;

/**
 * @author marcio.endo@objectos.com.br (Marcio Endo)
 */
class PageUIBuilderPojo
    implements
    PageUIBuilder,
    PageUIBuilder.PageTitleDoneBuilder,
    PageUIBuilder.BreadcrumbDoneBuilder,
    PageUIBuilder.ContextDoneBuilder {

  private final Injector injector;

  private final PageTitle pageTitle;
  private final Breadcrumb breadcrumb;

  private PageRequest request;

  private Context context;
  private Class<? extends AbstractPage> pageClass;

  @Inject
  public PageUIBuilderPojo(Injector injector, PageTitle pageTitle, Breadcrumb breadcrumb) {
    this.injector = injector;
    this.pageTitle = pageTitle;
    this.breadcrumb = breadcrumb;
  }

  @Override
  public PageTitleBuilder pageTitle() {
    return new PageTitleBuilderPojo();
  }

  @Override
  public BreadcrumbBuilder breadcrumb() {
    return new BreadcrumbBuilderPojo();
  }

  @Override
  public ContextBuilder context() {
    context = injector.getInstance(Context.class);
    return new ContextBuilderPojo();
  }

  @Override
  public PageUI show(AbstractPage page) {
    pageClass = page.getClass();
    return build();
  }

  PageUIBuilder after(Class<? extends AbstractPage> previousPageClass) {
    if (previousPageClass != null) {
      AbstractPage page = injector.getInstance(previousPageClass);

      PageUI ui = page.ui(request);
      PageUIPojo uiPojo = ui.toPojo();

      uiPojo.addAllTo(pageTitle);
      uiPojo.addAllTo(breadcrumb);
    }

    return this;
  }

  Injector getInjector() {
    return injector;
  }

  PageTitle getPageTitle() {
    return pageTitle;
  }

  Breadcrumb getBreadcrumb() {
    return breadcrumb;
  }

  Context getContext() {
    return context;
  }

  Class<? extends AbstractPage> getPageClass() {
    return pageClass;
  }

  PageUIBuilderPojo with(PageRequest request) {
    this.request = request;
    return this;
  }

  private PageUI build() {
    return new PageUIPojo(this);
  }

  private class PageTitleBuilderPojo
      implements
      PageUIBuilder.PageTitleBuilder,
      PageUIBuilder.PageTitleDisplayBuilder {

    @Override
    public PageTitleDisplayBuilder display(String text) {
      pageTitle.display(text);
      return this;
    }

    @Override
    public PageTitleDoneBuilder done() {
      return PageUIBuilderPojo.this;
    }

    @Override
    public PageTitleDoneBuilder withSeparator(String separator) {
      pageTitle.withSeparator(separator);
      return done();
    }

  }

  private class BreadcrumbBuilderPojo
      implements
      PageUIBuilder.BreadcrumbBuilder,
      PageUIBuilder.BreadcrumbDisplayBuilder {

    private String text;

    @Override
    public BreadcrumbDisplayBuilder display(String text) {
      this.text = text;
      return this;
    }

    @Override
    public BreadcrumbBuilder linkTo(String link) {
      String url = Strings.emptyToNull(link);
      breadcrumb.add(text, url);
      return this;
    }

    @Override
    public BreadcrumbDoneBuilder done() {
      return PageUIBuilderPojo.this;
    }

  }

  private class ContextBuilderPojo
      implements
      PageUIBuilder.ContextBuilder {

    @Override
    public ContextBuilder accept(ContextVisitor visitor) {
      visitor.visitContext(context);
      return this;
    }

    @Override
    public ContextBuilder put(String var, Object value) {
      context.put(var, value);
      return this;
    }

    @Override
    public ContextDoneBuilder done() {
      return PageUIBuilderPojo.this;
    }

  }

}