/*
 * Copyright 2014 Objectos, Fábrica de Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package br.com.objectos.way.ui;

import static com.google.common.collect.Lists.newArrayList;

import java.util.List;
import java.util.Objects;

import br.com.objectos.way.ui.form.Errors;
import br.com.objectos.way.ui.form.FormResponse.Error;

import com.google.common.base.Optional;
import com.google.common.base.Preconditions;

/**
 * @author marcio.endo@objectos.com.br (Marcio Endo)
 */
class UserInputConfigurePojo
    implements
    UserInputConfigure,
    UserInputConfigure.UserInputConfigureExecute,
    UserInputConfigure.UserInputConfigureRedirectTo {

  private final List<Error> errors = newArrayList();
  private final List<Error> formErrors = newArrayList();

  private UserInputAction action;
  private Context context;
  private String redirectUrl;

  @Override
  public UserInputConfigureAddMessage addMessage(final String message) {
    return new UserInputConfigureAddMessage() {

      @Override
      public UserInputConfigure toField(String fieldName) {
        errors.add(Errors.toField(fieldName, message));

        return UserInputConfigurePojo.this;
      }

      @Override
      public UserInputConfigure toForm() {
        formErrors.add(Errors.toForm(message));

        return UserInputConfigurePojo.this;
      }

    };
  }

  @Override
  public UserInputConfigureAddMessage addMessage(String template, Object... args) {
    String message = String.format(template, args);
    return addMessage(message);
  }

  @Override
  public UserInputConfigureExecute execute(UserInputAction action) {
    this.action = Preconditions.checkNotNull(action);
    return this;
  }

  @Override
  public UserInputConfigureRedirectTo redirectTo(String url) {
    Objects.requireNonNull(url, "url");
    redirectUrl = url;
    return this;
  }

  @Override
  public UserInput reply() {
    if (action != null) {
      tryToExecute();
    }

    return UserInput.builder()
        .context(Optional.fromNullable(context))
        .errors(errors)
        .formErrors(formErrors)
        .redirectUrl(Optional.fromNullable(redirectUrl))
        .build();
  }

  private void tryToExecute() {
    if (errors.isEmpty() && formErrors.isEmpty()) {
      try {
        action.execute();
      } catch (Throwable e) {
        formErrors.add(Errors.toForm(e));
      }
    }
  }

}