/*
 * Copyright 2011 Objectos, Fábrica de Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package br.com.objectos.way.ui.form;

import javax.validation.ConstraintViolation;

import br.com.objectos.way.ui.form.FormResponse.Error;

import com.google.common.base.Throwables;

/**
 * @author marcio.endo@objectos.com.br (Marcio Endo)
 */
public abstract class Errors implements FormResponse.Error {

  private final String message;

  private final String messageBody;

  Errors(ConstraintViolation<?> violation) {
    message = violation.getMessage();
    messageBody = "";
  }

  Errors(String message) {
    this.message = message;
    messageBody = "";
  }

  Errors(String message, String messageBody) {
    this.message = message;
    this.messageBody = messageBody;
  }

  Errors(Throwable t) {
    message = t.getMessage();
    messageBody = Throwables.getStackTraceAsString(t);
  }

  public static Error toForm(String message) {
    return new FormError(message);
  }

  public static Error toForm(Throwable t) {
    return new FormError(t);
  }

  public static Error toField(ConstraintViolation<?> violation) {
    return new FieldError(violation);
  }

  public static Error toField(String name, String message) {
    return new FieldError(name, message);
  }

  @Override
  public String getName() {
    return null;
  }

  @Override
  public String getMessage() {
    return message;
  }

  @Override
  public String getMessageBody() {
    return messageBody;
  }

  static class FormError extends Errors {

    private FormError(String message) {
      super(message);
    }

    private FormError(String message, String messageBody) {
      super(message, messageBody);
    }

    private FormError(Throwable t) {
      super(t);
    }

  }

  static class FieldError extends Errors {

    private final String name;

    private FieldError(ConstraintViolation<?> violation) {
      super(violation);
      name = violation.getPropertyPath().toString();
    }

    private FieldError(String name, String message) {
      super(message);
      this.name = name;
    }

    @Override
    public String getName() {
      return name;
    }

  }

}